<?php
/**
 * Cart Compact Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cart Compact Widget - Single column compact layout
 */
class Cart_Compact extends AI_Widget_Base {

    public function get_name() {
        return 'ai_cart_compact';
    }

    public function get_title() {
        return __('Cart Compact', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-cart';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control('title', array('label' => __('Title', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));
        $this->add_control('checkout_text', array('label' => __('Checkout Button', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        
        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        // Check if WooCommerce cart is initialized (may be null during REST API requests)
        if (!WC() || !WC()->cart) {
            return;
        }

        $cart = WC()->cart;
        $cart_items = $cart->get_cart();
        ?>
        <section class="py-8 bg-[var(--bg-secondary)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="cart" data-block-variant="cart-compact">
            <div class="max-w-2xl mx-auto px-4">
                <h1 class="text-2xl font-bold text-[var(--text-primary)] mb-6"><?php echo esc_html($settings['title']); ?></h1>

                <?php if (empty($cart_items)) : ?>
                    <div class="text-center py-12">
                        <svg class="w-16 h-16 mx-auto text-[var(--text-muted)] mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                        <p class="text-[var(--text-muted)]"><?php _e('Your cart is empty', 'ai-theme'); ?></p>
                    </div>
                <?php else : ?>
                    <!-- Items -->
                    <div class="space-y-3 mb-6">
                        <?php foreach ($cart_items as $cart_item_key => $cart_item) : 
                            $product = $cart_item['data'];
                            $image = wp_get_attachment_image_url($product->get_image_id(), 'thumbnail');
                        ?>
                            <div class="bg-[var(--bg-primary)] rounded-[var(--card-radius,12px)] p-4 shadow-sm">
                                <div class="flex gap-4">
                                    <?php if ($image) : ?>
                                        <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-20 h-20 object-cover rounded-[var(--img-radius,8px)]" />
                                    <?php endif; ?>
                                    <div class="flex-1">
                                        <div class="flex justify-between items-start">
                                            <h3 class="font-semibold text-[var(--text-primary)]"><?php echo esc_html($product->get_name()); ?></h3>
                                            <a href="<?php echo esc_url(wc_get_cart_remove_url($cart_item_key)); ?>" class="p-1 text-[var(--text-muted)] hover:text-red-500">
                                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>
                                            </a>
                                        </div>
                                        <div class="flex justify-between items-center mt-3">
                                            <span class="text-[var(--text-secondary)]">x<?php echo esc_html($cart_item['quantity']); ?></span>
                                            <span class="font-bold text-[var(--text-primary)]"><?php echo wc_price($product->get_price() * $cart_item['quantity']); ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <!-- Summary -->
                    <div class="bg-[var(--bg-primary)] rounded-[var(--card-radius,12px)] p-6 shadow-sm">
                        <div class="flex justify-between mb-2 text-[var(--text-secondary)]">
                            <span><?php _e('Subtotal', 'ai-theme'); ?></span>
                            <span><?php echo wc_price($cart->get_subtotal()); ?></span>
                        </div>
                        <div class="flex justify-between text-lg font-bold text-[var(--text-primary)] pt-3 border-t border-[var(--border-color)]">
                            <span><?php _e('Total', 'ai-theme'); ?></span>
                            <span><?php echo wc_price($cart->get_total('')); ?></span>
                        </div>
                        <a href="<?php echo esc_url(wc_get_checkout_url()); ?>" class="block w-full mt-4 py-3 text-center bg-[var(--primary)] text-white rounded-[var(--btn-radius,8px)] font-semibold hover:opacity-90">
                            <?php echo esc_html($settings['checkout_text']); ?>
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </section>
        <?php
    }
}
