<?php
/**
 * Cart Minimal Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cart Minimal Widget - Ultra-minimal cart
 */
class Cart_Minimal extends AI_Widget_Base {

    public function get_name() {
        return 'ai_cart_minimal';
    }

    public function get_title() {
        return __('Cart Minimal', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-cart';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control('title', array('label' => __('Title', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));
        $this->add_control('checkout_text', array('label' => __('Checkout Button', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        
        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        // Check if WooCommerce cart is initialized (may be null during REST API requests)
        if (!WC() || !WC()->cart) {
            return;
        }

        $cart = WC()->cart;
        $cart_items = $cart->get_cart();
        ?>
        <section class="py-8 bg-[var(--bg-primary)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="cart" data-block-variant="cart-minimal">
            <div class="max-w-2xl mx-auto px-4">
                <h2 class="text-xl font-bold text-[var(--text-primary)] mb-6"><?php echo esc_html($settings['title']); ?></h2>

                <?php if (empty($cart_items)) : ?>
                    <div class="py-12 text-center text-[var(--text-muted)]">
                        <?php _e('Cart is empty', 'ai-theme'); ?>
                    </div>
                <?php else : ?>
                    <div class="space-y-3 mb-6">
                        <?php foreach ($cart_items as $cart_item_key => $cart_item) : 
                            $product = $cart_item['data'];
                            $image = wp_get_attachment_image_url($product->get_image_id(), 'thumbnail');
                        ?>
                            <div class="flex items-center gap-4 py-3 border-b border-[var(--border-color)]">
                                <?php if ($image) : ?>
                                    <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-12 h-12 object-cover rounded" />
                                <?php endif; ?>
                                <div class="flex-1 min-w-0">
                                    <p class="font-medium text-[var(--text-primary)] truncate"><?php echo esc_html($product->get_name()); ?></p>
                                    <p class="text-sm text-[var(--text-muted)]"><?php echo wc_price($product->get_price()); ?></p>
                                </div>
                                <span class="w-8 text-center text-sm"><?php echo esc_html($cart_item['quantity']); ?></span>
                                <p class="font-semibold text-[var(--text-primary)] w-20 text-right">
                                    <?php echo wc_price($product->get_price() * $cart_item['quantity']); ?>
                                </p>
                                <a href="<?php echo esc_url(wc_get_cart_remove_url($cart_item_key)); ?>" class="p-1 text-[var(--text-muted)] hover:text-red-500">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>
                                </a>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <div class="flex items-center justify-between pt-4">
                        <span class="text-lg font-semibold text-[var(--text-primary)]"><?php _e('Total', 'ai-theme'); ?></span>
                        <span class="text-2xl font-bold text-[var(--text-primary)]"><?php echo wc_price($cart->get_total('')); ?></span>
                    </div>
                    <a href="<?php echo esc_url(wc_get_checkout_url()); ?>" class="block w-full mt-4 bg-[var(--primary)] text-white py-3 text-center rounded-[var(--btn-radius)] font-semibold hover:opacity-90">
                        <?php echo esc_html($settings['checkout_text']); ?>
                    </a>
                <?php endif; ?>
            </div>
        </section>
        <?php
    }
}
