<?php
/**
 * Cart Table Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cart Table Widget - Classic table layout
 */
class Cart_Table extends AI_Widget_Base {

    public function get_name() {
        return 'ai_cart_table';
    }

    public function get_title() {
        return __('Cart Table', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-cart';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control('title', array('label' => __('Title', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));
        $this->add_control('checkout_text', array('label' => __('Checkout Button', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        
        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        // Check if WooCommerce cart is initialized (may be null during REST API requests)
        if (!WC() || !WC()->cart) {
            return;
        }

        $cart = WC()->cart;
        $cart_items = $cart->get_cart();
        ?>
        <section class="py-12 bg-[var(--bg-primary)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="cart" data-block-variant="cart-table">
            <div class="max-w-6xl mx-auto px-4">
                <h1 class="text-2xl font-bold text-[var(--text-primary)] mb-8"><?php echo esc_html($settings['title']); ?></h1>

                <?php if (empty($cart_items)) : ?>
                    <div class="py-16 text-center">
                        <p class="text-xl text-[var(--text-muted)] mb-4"><?php _e('Your cart is empty', 'ai-theme'); ?></p>
                        <a href="<?php echo esc_url(wc_get_page_permalink('shop')); ?>" class="text-[var(--primary)] hover:underline">
                            <?php _e('Start shopping', 'ai-theme'); ?>
                        </a>
                    </div>
                <?php else : ?>
                    <!-- Table -->
                    <div class="bg-[var(--bg-primary)] border border-[var(--border-color)] rounded-[var(--card-radius)] overflow-hidden mb-8">
                        <table class="w-full">
                            <thead class="bg-[var(--bg-secondary)]">
                                <tr>
                                    <th class="text-left px-6 py-4 text-sm font-semibold text-[var(--text-secondary)]"><?php _e('Product', 'ai-theme'); ?></th>
                                    <th class="text-center px-6 py-4 text-sm font-semibold text-[var(--text-secondary)]"><?php _e('Price', 'ai-theme'); ?></th>
                                    <th class="text-center px-6 py-4 text-sm font-semibold text-[var(--text-secondary)]"><?php _e('Quantity', 'ai-theme'); ?></th>
                                    <th class="text-right px-6 py-4 text-sm font-semibold text-[var(--text-secondary)]"><?php _e('Total', 'ai-theme'); ?></th>
                                    <th class="px-6 py-4"></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($cart_items as $cart_item_key => $cart_item) : 
                                    $product = $cart_item['data'];
                                    $image = wp_get_attachment_image_url($product->get_image_id(), 'thumbnail');
                                ?>
                                    <tr class="border-t border-[var(--border-color)]">
                                        <td class="px-6 py-4">
                                            <div class="flex items-center gap-4">
                                                <?php if ($image) : ?>
                                                    <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-16 h-16 object-cover rounded" />
                                                <?php endif; ?>
                                                <div>
                                                    <p class="font-medium text-[var(--text-primary)]"><?php echo esc_html($product->get_name()); ?></p>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 text-center text-[var(--text-secondary)]">
                                            <?php echo wc_price($product->get_price()); ?>
                                        </td>
                                        <td class="px-6 py-4">
                                            <div class="flex items-center justify-center">
                                                <span class="w-10 text-center font-medium"><?php echo esc_html($cart_item['quantity']); ?></span>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 text-right font-semibold text-[var(--text-primary)]">
                                            <?php echo wc_price($product->get_price() * $cart_item['quantity']); ?>
                                        </td>
                                        <td class="px-6 py-4 text-right">
                                            <a href="<?php echo esc_url(wc_get_cart_remove_url($cart_item_key)); ?>" class="p-2 text-[var(--text-muted)] hover:text-red-500 hover:bg-red-50 rounded inline-flex">
                                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- Summary -->
                    <div class="flex justify-end">
                        <div class="w-full max-w-sm bg-[var(--bg-secondary)] rounded-[var(--card-radius)] p-6">
                            <div class="space-y-3 mb-4">
                                <div class="flex justify-between text-[var(--text-secondary)]">
                                    <span><?php _e('Subtotal', 'ai-theme'); ?></span>
                                    <span><?php echo wc_price($cart->get_subtotal()); ?></span>
                                </div>
                                <?php if ($cart->get_shipping_total() > 0) : ?>
                                    <div class="flex justify-between text-[var(--text-secondary)]">
                                        <span><?php _e('Shipping', 'ai-theme'); ?></span>
                                        <span><?php echo wc_price($cart->get_shipping_total()); ?></span>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="flex justify-between pt-4 border-t border-[var(--border-color)]">
                                <span class="text-lg font-semibold text-[var(--text-primary)]"><?php _e('Total', 'ai-theme'); ?></span>
                                <span class="text-xl font-bold text-[var(--text-primary)]"><?php echo wc_price($cart->get_total('')); ?></span>
                            </div>
                            <a href="<?php echo esc_url(wc_get_checkout_url()); ?>" class="block w-full mt-6 bg-[var(--primary)] text-white py-3 text-center rounded-[var(--btn-radius)] font-semibold hover:opacity-90">
                                <?php echo esc_html($settings['checkout_text']); ?>
                            </a>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </section>
        <?php
    }
}
