<?php
/**
 * Footer V1 Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Footer V1 Widget - Classic two-row footer
 */
class Footer_V1 extends AI_Widget_Base {

    public function get_name() {
        return 'ai_footer_v1';
    }

    public function get_title() {
        return __('Footer V1', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-footer';
    }

    public function get_categories() {
        return array('ai-theme-footer');
    }

    public function get_keywords() {
        return array('footer', 'bottom', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'logo_text',
            array(
                'label'   => __('Logo Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'logo_image',
            array(
                'label' => __('Logo Image', 'ai-theme'),
                'type'  => Controls_Manager::MEDIA,
            )
        );

        $this->add_control(
            'description',
            array(
                'label'   => __('Description', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $this->add_control(
            'copyright',
            array(
                'label'   => __('Copyright Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'platform',
            array(
                'label'   => __('Platform', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'facebook',
                'options' => array(
                    'facebook'  => 'Facebook',
                    'twitter'   => 'Twitter/X',
                    'instagram' => 'Instagram',
                    'linkedin'  => 'LinkedIn',
                    'youtube'   => 'YouTube',
                    'telegram'  => 'Telegram',
                ),
            )
        );

        $repeater->add_control(
            'link',
            array(
                'label' => __('Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $this->add_control(
            'social_links',
            array(
                'label'  => __('Social Links', 'ai-theme'),
                'type'   => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $logo_url = !empty($settings['logo_image']['url']) ? $settings['logo_image']['url'] : '';
        // Fallback to company_name for backward compatibility
        $brand_name = !empty($settings['logo_text']) ? $settings['logo_text'] : ($settings['company_name'] ?? '');
        ?>
        <footer
            class="ai-footer bg-[var(--bg-dark,#111827)] text-[var(--text-on-dark)]"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="footer"
            data-block-variant="footer-v1"
        >
            <div class="max-w-7xl mx-auto px-6 py-12">
                <div class="flex flex-col md:flex-row justify-between items-start md:items-center gap-8">
                    <!-- Brand -->
                    <div class="max-w-md">
                        <a href="<?php echo esc_url(home_url('/')); ?>" class="inline-block mb-3">
                            <?php if ($logo_url) : ?>
                                <img src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr($brand_name); ?>" class="h-8 object-contain brightness-0 invert" />
                            <?php else : ?>
                                <h3 class="text-2xl font-bold"><?php echo esc_html($brand_name); ?></h3>
                            <?php endif; ?>
                        </a>
                        <?php if (!empty($settings['description'])) : ?>
                            <p class="text-[var(--text-muted,#9ca3af)]"><?php echo esc_html($settings['description']); ?></p>
                        <?php endif; ?>
                    </div>

                    <!-- Social Links -->
                    <?php if (!empty($settings['social_links'])) : ?>
                        <div class="flex gap-3">
                            <?php foreach ($settings['social_links'] as $social) : ?>
                                <a
                                    href="<?php echo esc_url($social['link']['url'] ?? '#'); ?>"
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    class="w-10 h-10 bg-white/10 rounded-full flex items-center justify-center hover:bg-[var(--primary-on-dark,var(--primary))] transition-colors"
                                    aria-label="<?php echo esc_attr($social['platform']); ?>"
                                >
                                    <?php echo $this->get_social_icon_svg($social['platform']); ?>
                                </a>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Bottom Bar -->
            <div class="border-t border-white/10">
                <div class="max-w-7xl mx-auto px-6 py-6">
                    <div class="flex flex-col md:flex-row justify-between items-center gap-4">
                        <p class="text-sm text-[var(--text-muted,#9ca3af)]"><?php echo esc_html($settings['copyright']); ?></p>
                        <div class="flex gap-6 text-sm">
                            <a href="<?php echo esc_url(home_url('/privacy-policy')); ?>" class="text-[var(--text-muted,#9ca3af)] hover:text-[var(--link-color-on-dark,#ffffff)] transition-colors">
                                <?php echo esc_html(ai_get_text('navigation.labels.privacy', 'Privacy Policy')); ?>
                            </a>
                            <a href="<?php echo esc_url(home_url('/cookies')); ?>" class="text-[var(--text-muted,#9ca3af)] hover:text-[var(--link-color-on-dark,#ffffff)] transition-colors">
                                <?php echo esc_html(ai_get_text('navigation.labels.cookies', 'Cookie Policy')); ?>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </footer>
        <?php
    }

    private function get_social_icon_svg($platform) {
        $icons = array(
            'facebook'  => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>',
            'twitter'   => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24"><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/></svg>',
            'instagram' => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24"><path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z"/></svg>',
            'linkedin'  => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24"><path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/></svg>',
            'youtube'   => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24"><path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z"/></svg>',
            'telegram'  => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 24 24"><path d="M11.944 0A12 12 0 0 0 0 12a12 12 0 0 0 12 12 12 12 0 0 0 12-12A12 12 0 0 0 12 0a12 12 0 0 0-.056 0zm4.962 7.224c.1-.002.321.023.465.14a.506.506 0 0 1 .171.325c.016.093.036.306.02.472-.18 1.898-.962 6.502-1.36 8.627-.168.9-.499 1.201-.82 1.23-.696.065-1.225-.46-1.9-.902-1.056-.693-1.653-1.124-2.678-1.8-1.185-.78-.417-1.21.258-1.91.177-.184 3.247-2.977 3.307-3.23.007-.032.014-.15-.056-.212s-.174-.041-.249-.024c-.106.024-1.793 1.14-5.061 3.345-.48.33-.913.49-1.302.48-.428-.008-1.252-.241-1.865-.44-.752-.245-1.349-.374-1.297-.789.027-.216.325-.437.893-.663 3.498-1.524 5.83-2.529 6.998-3.014 3.332-1.386 4.025-1.627 4.476-1.635z"/></svg>',
        );
        return $icons[$platform] ?? '';
    }
}
