<?php
/**
 * Hero Cards Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Hero Cards Widget - Hero with feature cards
 */
class Hero_Cards extends AI_Widget_Base {

    public function get_name() {
        return 'ai_hero_cards';
    }

    public function get_title() {
        return __('Hero Cards', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-banner';
    }

    public function get_categories() {
        return array('ai-theme-hero', 'ai-theme-blocks');
    }

    public function get_keywords() {
        return array('hero', 'banner', 'cards', 'features', 'ai', 'landing');
    }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'       => __('Heading', 'ai-theme'),
                'type'        => Controls_Manager::TEXT,
                'default'     => '',
                'label_block' => true,
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'   => __('Subheading', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
                'rows'    => 3,
            )
        );

        $this->add_control(
            'cta_text',
            array(
                'label'   => __('Button Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'cta_link',
            array(
                'label'   => __('Button Link', 'ai-theme'),
                'type'    => Controls_Manager::URL,
                'default' => array('url' => '#'),
            )
        );

        $this->end_controls_section();

        // Features Section
        $this->start_controls_section(
            'features_section',
            array(
                'label' => __('Features', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'title',
            array(
                'label'   => __('Title', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'description',
            array(
                'label'   => __('Description', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $repeater->add_control(
            'icon',
            array(
                'label'   => __('Icon', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'zap',
                'options' => array(
                    'zap'        => __('Lightning', 'ai-theme'),
                    'shield'     => __('Shield', 'ai-theme'),
                    'clock'      => __('Clock', 'ai-theme'),
                    'headphones' => __('Support', 'ai-theme'),
                ),
            )
        );

        $this->add_control(
            'features',
            array(
                'label'   => __('Features', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(
                    array('title' => 'Fast', 'description' => 'Lightning speed', 'icon' => 'zap'),
                    array('title' => 'Secure', 'description' => 'Protected data', 'icon' => 'shield'),
                    array('title' => 'Quick Setup', 'description' => 'Minutes not weeks', 'icon' => 'clock'),
                    array('title' => 'Support', 'description' => '24/7 help', 'icon' => 'headphones'),
                ),
            )
        );

        $this->end_controls_section();

        // Layout
        $this->start_controls_section(
            'layout_section',
            array(
                'label' => __('Layout', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_theme_style_control();

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        $theme = $settings['theme_style'] ?? 'inherit';
        if ($theme === 'inherit') {
            $theme = get_option('ai_theme_style', 'modern');
        }

        $features = $settings['features'] ?? array();
        // Use theme primary color with variations
        $icon_wrapper_class = 'bg-[var(--primary-light)] text-[var(--primary)]';

        $icons = array(
            'zap' => '<svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z" /></svg>',
            'shield' => '<svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" /></svg>',
            'clock' => '<svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" /></svg>',
            'headphones' => '<svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" /></svg>',
        );
        ?>
        <div data-theme="<?php echo esc_attr($theme); ?>" class="ai-theme-block">
            <section
                class="ai-hero ai-hero-cards section-light py-20 lg:py-32"
                data-block-type="hero"
                data-block-variant="hero-cards"
            >
                <div class="max-w-7xl mx-auto px-6 md:px-8">
                    <!-- Header -->
                    <div class="text-center max-w-3xl mx-auto mb-16">
                        <?php if (!empty($settings['heading'])) : ?>
                            <h1 class="ai-hero-heading text-4xl md:text-5xl lg:text-6xl font-bold text-[var(--text-primary)] leading-tight mb-6">
                                <?php echo esc_html($settings['heading']); ?>
                            </h1>
                        <?php endif; ?>

                        <?php if (!empty($settings['subheading'])) : ?>
                            <p class="ai-hero-subheading text-lg md:text-xl text-[var(--text-secondary)] mb-8">
                                <?php echo esc_html($settings['subheading']); ?>
                            </p>
                        <?php endif; ?>

                        <?php if (!empty($settings['cta_text'])) : ?>
                            <a
                                href="<?php echo esc_url($settings['cta_link']['url'] ?? '#'); ?>"
                                class="inline-flex items-center gap-2 px-8 py-4 bg-[var(--primary)] text-[var(--text-on-primary)] font-semibold rounded-lg hover:opacity-90 transition-colors shadow-lg"
                            >
                                <?php echo esc_html($settings['cta_text']); ?>
                                <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7l5 5m0 0l-5 5m5-5H6" />
                                </svg>
                            </a>
                        <?php endif; ?>
                    </div>

                    <!-- Feature Cards -->
                    <?php if (!empty($features)) : ?>
                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                            <?php foreach (array_slice($features, 0, 4) as $idx => $feature) : ?>
                                <div class="bg-[var(--bg-secondary)] rounded-2xl p-6 hover:bg-[var(--bg-tertiary)] transition-colors group">
                                    <div class="w-12 h-12 rounded-xl <?php echo $icon_wrapper_class; ?> flex items-center justify-center mb-4 group-hover:scale-110 transition-transform">
                                        <?php echo $icons[$feature['icon']] ?? $icons['zap']; ?>
                                    </div>
                                    <h3 class="text-lg font-semibold text-[var(--text-primary)] mb-2">
                                        <?php echo esc_html($feature['title']); ?>
                                    </h3>
                                    <p class="text-[var(--text-secondary)] text-sm">
                                        <?php echo esc_html($feature['description']); ?>
                                    </p>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </section>
        </div>
        <?php
    }
}
