<?php
/**
 * Navbar Floating Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Navbar Floating Widget - Плавающий навбар с glassmorphism
 */
class Navbar_Floating extends AI_Widget_Base {

    public function get_name() {
        return 'ai_navbar_floating';
    }

    public function get_title() {
        return __('Navbar Floating', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-nav-menu';
    }

    public function get_categories() {
        return array('ai-theme-navigation');
    }

    public function get_keywords() {
        return array('navbar', 'navigation', 'menu', 'header', 'floating', 'glassmorphism', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'logo_text',
            array(
                'label'   => __('Logo Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'logo_image',
            array(
                'label' => __('Logo Image', 'ai-theme'),
                'type'  => Controls_Manager::MEDIA,
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'title',
            array(
                'label'   => __('Title', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'link',
            array(
                'label' => __('Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $this->add_control(
            'menu_items',
            array(
                'label'   => __('Menu Items', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(
                    array('title' => 'Home', 'link' => array('url' => '/')),
                    array('title' => 'About', 'link' => array('url' => '/about')),
                    array('title' => 'Services', 'link' => array('url' => '/services')),
                    array('title' => 'Contact', 'link' => array('url' => '/contact')),
                ),
            )
        );

        $this->add_control(
            'show_cart',
            array(
                'label'        => __('Show Cart', 'ai-theme'),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __('Yes', 'ai-theme'),
                'label_off'    => __('No', 'ai-theme'),
                'return_value' => 'yes',
                'default'      => '',
            )
        );

        $this->add_control(
            'show_sign_in',
            array(
                'label'        => __('Show Sign In', 'ai-theme'),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __('Yes', 'ai-theme'),
                'label_off'    => __('No', 'ai-theme'),
                'return_value' => 'yes',
                'default'      => '',
            )
        );

        $this->add_control(
            'cta_text',
            array(
                'label'   => __('CTA Button Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'cta_link',
            array(
                'label' => __('CTA Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $unique_id = 'navbar-floating-' . $this->get_id();

        // Handle logo_image
        $logo_url = '';
        if (!empty($settings['logo_image'])) {
            if (is_string($settings['logo_image'])) {
                $logo_url = $settings['logo_image'];
            } elseif (is_array($settings['logo_image'])) {
                $logo_url = $settings['logo_image']['url'] ?? '';
            }
        }

        // WooCommerce
        $cart_url = function_exists('wc_get_cart_url') ? wc_get_cart_url() : home_url('/cart/');
        $account_url = function_exists('wc_get_page_permalink') ? wc_get_page_permalink('myaccount') : home_url('/my-account/');
        $cart_count = 0;
        if (function_exists('WC') && WC()->cart) {
            $cart_count = WC()->cart->get_cart_contents_count();
        }

        // User state
        $is_logged_in = is_user_logged_in();
        $current_user = wp_get_current_user();
        ?>
        <header class="ai-navbar ai-navbar-floating sticky top-0 z-50 pt-4 px-4" id="<?php echo esc_attr($unique_id); ?>" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="navbar" data-block-variant="navbar-floating">
            <nav class="max-w-5xl mx-auto">
                <!-- Floating container with glassmorphism -->
                <div class="bg-[var(--bg-primary)]/80 backdrop-blur-xl border border-[var(--border-color)] rounded-2xl shadow-lg shadow-black/5">
                    <div class="flex items-center justify-between h-14 px-4">
                        <!-- Logo -->
                        <a href="<?php echo esc_url(home_url('/')); ?>" class="flex items-center gap-2 flex-shrink-0">
                            <?php if (!empty($logo_url)) : ?>
                                <img src="<?php echo esc_url($logo_url); ?>" alt="Logo" class="h-8 object-contain">
                            <?php endif; ?>
                            <?php if (empty($logo_url) && !empty($settings['logo_text'])) : ?>
                                <div class="w-8 h-8 bg-gradient-to-br from-[var(--primary)] to-[var(--secondary)] rounded-lg flex items-center justify-center text-white font-bold text-sm">
                                    <?php echo esc_html(mb_substr($settings['logo_text'], 0, 1)); ?>
                                </div>
                            <?php endif; ?>
                            <?php if (!empty($settings['logo_text'])) : ?>
                                <span class="text-lg font-semibold text-[var(--text-primary)] hidden sm:block"><?php echo esc_html($settings['logo_text']); ?></span>
                            <?php endif; ?>
                        </a>

                        <!-- Desktop Menu - pill style -->
                        <div class="hidden lg:flex items-center">
                            <div class="flex items-center bg-[var(--bg-secondary)] rounded-full px-1 py-1">
                                <?php $first = true; ?>
                                <?php foreach (($settings['menu_items'] ?? []) as $item) :
                                    $link_url = '#';
                                    if (!empty($item['link'])) {
                                        if (is_string($item['link'])) {
                                            $link_url = $item['link'];
                                        } elseif (is_array($item['link'])) {
                                            $link_url = $item['link']['url'] ?? $item['link']['href'] ?? '#';
                                        }
                                    }
                                    $is_active = $first;
                                    $first = false;
                                ?>
                                    <a
                                        href="<?php echo esc_url($link_url); ?>"
                                        class="px-4 py-1.5 rounded-full text-sm font-medium transition-all <?php echo $is_active ? 'bg-[var(--bg-primary)] text-[var(--text-primary)] shadow-sm' : 'text-[var(--text-secondary)] hover:text-[var(--text-primary)]'; ?>"
                                    >
                                        <?php echo esc_html($item['title'] ?? ''); ?>
                                    </a>
                                <?php endforeach; ?>
                            </div>
                        </div>

                        <!-- Actions -->
                        <div class="flex items-center gap-2">
                            <?php if ($settings['show_cart'] === 'yes') : ?>
                                <!-- Cart -->
                                <a href="<?php echo esc_url($cart_url); ?>" class="relative p-2 rounded-full hover:bg-[var(--bg-secondary)] transition-colors">
                                    <svg class="w-5 h-5 text-[var(--text-secondary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                    </svg>
                                    <?php if ($cart_count > 0) : ?>
                                    <span class="absolute -top-0.5 -right-0.5 w-4 h-4 bg-[var(--primary)] text-white text-[10px] font-bold rounded-full flex items-center justify-center"><?php echo esc_html($cart_count); ?></span>
                                    <?php endif; ?>
                                </a>
                            <?php endif; ?>

                            <?php if ($settings['show_sign_in'] === 'yes') : ?>
                                <!-- User/Profile -->
                                <a href="<?php echo esc_url($account_url); ?>" class="hidden sm:flex items-center gap-2 px-2 py-1 hover:bg-[var(--bg-secondary)] rounded-lg transition-colors">
                                    <div class="w-7 h-7 bg-[var(--primary-light)] rounded-full flex items-center justify-center text-[var(--primary)] text-sm font-semibold">
                                        <?php echo $is_logged_in ? esc_html(mb_strtoupper(mb_substr($current_user->display_name ?: $current_user->user_login, 0, 1))) : 'A'; ?>
                                    </div>
                                </a>
                            <?php endif; ?>

                            <?php if (!empty($settings['cta_text']) && $settings['show_cart'] !== 'yes') :
                                $cta_url = !empty($settings['cta_link']['url']) ? $settings['cta_link']['url'] : '#';
                            ?>
                                <!-- CTA Button -->
                                <a href="<?php echo esc_url($cta_url); ?>" class="hidden md:block px-4 py-1.5 bg-[var(--primary)] text-white text-sm font-medium rounded-full hover:opacity-90 transition-opacity">
                                    <?php echo esc_html($settings['cta_text']); ?>
                                </a>
                            <?php endif; ?>

                            <!-- Mobile menu button -->
                            <button class="lg:hidden p-2 rounded-full hover:bg-[var(--bg-secondary)] ai-mobile-menu-toggle" aria-label="Toggle menu">
                                <svg class="w-5 h-5 text-[var(--text-primary)] ai-menu-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                                </svg>
                                <svg class="w-5 h-5 text-[var(--text-primary)] ai-close-icon hidden" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                </svg>
                            </button>
                        </div>
                    </div>

                    <!-- Mobile Menu -->
                    <div class="ai-mobile-menu lg:hidden hidden border-t border-[var(--border-color)] px-4 py-4">
                        <div class="space-y-1">
                            <?php foreach (($settings['menu_items'] ?? []) as $item) :
                                $link_url = '#';
                                if (!empty($item['link'])) {
                                    if (is_string($item['link'])) {
                                        $link_url = $item['link'];
                                    } elseif (is_array($item['link'])) {
                                        $link_url = $item['link']['url'] ?? $item['link']['href'] ?? '#';
                                    }
                                }
                            ?>
                                <a
                                    href="<?php echo esc_url($link_url); ?>"
                                    class="block px-4 py-2.5 rounded-xl text-sm font-medium text-[var(--text-secondary)] hover:bg-[var(--bg-secondary)] hover:text-[var(--text-primary)] transition-colors"
                                >
                                    <?php echo esc_html($item['title'] ?? ''); ?>
                                </a>
                            <?php endforeach; ?>
                        </div>
                        <?php if (!empty($settings['cta_text'])) :
                            $cta_url = !empty($settings['cta_link']['url']) ? $settings['cta_link']['url'] : '#';
                        ?>
                            <a href="<?php echo esc_url($cta_url); ?>" class="block mt-4 px-4 py-2.5 bg-[var(--primary)] text-white text-center text-sm font-medium rounded-xl hover:opacity-90 transition-opacity">
                                <?php echo esc_html($settings['cta_text']); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </nav>
        </header>

        <script>
        document.addEventListener('DOMContentLoaded', function() {
            var navbar = document.getElementById('<?php echo esc_js($unique_id); ?>');
            if (navbar) {
                var toggle = navbar.querySelector('.ai-mobile-menu-toggle');
                var menu = navbar.querySelector('.ai-mobile-menu');
                var menuIcon = navbar.querySelector('.ai-menu-icon');
                var closeIcon = navbar.querySelector('.ai-close-icon');

                if (toggle && menu) {
                    toggle.addEventListener('click', function() {
                        menu.classList.toggle('hidden');
                        menuIcon.classList.toggle('hidden');
                        closeIcon.classList.toggle('hidden');
                    });
                }
            }
        });
        </script>
        <?php
    }
}
