<?php
/**
 * Reviews Cards Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Reviews Cards Widget - Card grid layout
 */
class Reviews_Cards extends AI_Widget_Base {

    public function get_name() {
        return 'ai_reviews_cards';
    }

    public function get_title() {
        return __('Reviews Cards', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-review';
    }

    public function get_categories() {
        return array('ai-theme-testimonials');
    }

    protected function register_controls() {
        $this->start_controls_section('content_section', array('label' => __('Content', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT));

        $this->add_control('heading', array('label' => __('Heading', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));
        $this->add_control('subheading', array('label' => __('Subheading', 'ai-theme'), 'type' => Controls_Manager::TEXTAREA, 'default' => ''));

        $this->end_controls_section();

        // Reviews
        $this->start_controls_section('reviews_section', array('label' => __('Reviews', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT));

        $repeater = new Repeater();
        $repeater->add_control('author', array('label' => __('Author', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => 'John Doe'));
        $repeater->add_control('comment', array('label' => __('Comment', 'ai-theme'), 'type' => Controls_Manager::TEXTAREA, 'default' => 'Great product!'));
        $repeater->add_control('rating', array('label' => __('Rating', 'ai-theme'), 'type' => Controls_Manager::NUMBER, 'min' => 1, 'max' => 5, 'default' => 5));
        $repeater->add_control('avatar', array('label' => __('Avatar', 'ai-theme'), 'type' => Controls_Manager::MEDIA));
        $repeater->add_control('verified', array('label' => __('Verified', 'ai-theme'), 'type' => Controls_Manager::SWITCHER));

        $this->add_control('reviews', array(
            'label' => __('Reviews', 'ai-theme'),
            'type' => Controls_Manager::REPEATER,
            'fields' => $repeater->get_controls(),
            'default' => array(
                array('author' => 'Jane Smith', 'comment' => 'Excellent service and quality!', 'rating' => 5),
                array('author' => 'Mike Johnson', 'comment' => 'Very satisfied with my purchase.', 'rating' => 5),
                array('author' => 'Sarah Williams', 'comment' => 'Highly recommended!', 'rating' => 4),
            ),
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        ?>
        <section class="py-20 bg-[var(--bg-secondary)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="reviews" data-block-variant="reviews-cards">
            <div class="max-w-6xl mx-auto px-6">
                <div class="text-center mb-12">
                    <?php if (!empty($settings['heading'])) : ?>
                        <h2 class="text-3xl md:text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($settings['heading']); ?></h2>
                    <?php endif; ?>
                    <?php if (!empty($settings['subheading'])) : ?>
                        <p class="text-lg text-[var(--text-secondary)] max-w-2xl mx-auto"><?php echo esc_html($settings['subheading']); ?></p>
                    <?php endif; ?>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    <?php foreach ($settings['reviews'] as $review) : ?>
                        <div class="bg-[var(--bg-primary)] rounded-[var(--card-radius,16px)] p-6 shadow-[var(--shadow-card)] hover:shadow-xl transition-shadow duration-300">
                            <!-- Quote -->
                            <svg class="w-8 h-8 text-[var(--primary)] opacity-30 mb-4" fill="currentColor" viewBox="0 0 24 24"><path d="M14.017 21v-7.391c0-5.704 3.731-9.57 8.983-10.609l.995 2.151c-2.432.917-3.995 3.638-3.995 5.849h4v10h-9.983zm-14.017 0v-7.391c0-5.704 3.748-9.57 9-10.609l.996 2.151c-2.433.917-3.996 3.638-3.996 5.849h3.983v10h-9.983z"/></svg>

                            <p class="text-[var(--text-secondary)] mb-6 leading-relaxed line-clamp-4"><?php echo esc_html($review['comment']); ?></p>

                            <!-- Rating -->
                            <div class="flex items-center gap-1 mb-4">
                                <?php for ($i = 1; $i <= 5; $i++) : ?>
                                    <svg class="w-4 h-4 <?php echo $i <= $review['rating'] ? 'text-yellow-400 fill-yellow-400' : 'text-[var(--border-color)]'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                <?php endfor; ?>
                            </div>

                            <!-- Author -->
                            <div class="flex items-center gap-3">
                                <?php if (!empty($review['avatar']['url'])) : ?>
                                    <img src="<?php echo esc_url($review['avatar']['url']); ?>" alt="<?php echo esc_attr($review['author']); ?>" class="w-10 h-10 rounded-full object-cover" />
                                <?php else : ?>
                                    <div class="w-10 h-10 rounded-full bg-[var(--primary-light,#dbeafe)] flex items-center justify-center">
                                        <span class="text-[var(--primary)] font-semibold"><?php echo esc_html(substr($review['author'], 0, 1)); ?></span>
                                    </div>
                                <?php endif; ?>
                                <div>
                                    <p class="font-semibold text-[var(--text-primary)]"><?php echo esc_html($review['author']); ?></p>
                                    <?php if (!empty($review['verified']) && $review['verified'] === 'yes') : ?>
                                        <p class="text-xs text-[var(--primary)]"><?php _e('Verified', 'ai-theme'); ?></p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php
    }
}
