<?php
/**
 * Elementor Widgets Loader
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Widgets Loader Class
 */
class Widgets_Loader {

    /**
     * Instance
     *
     * @var Widgets_Loader|null
     */
    private static $instance = null;

    /**
     * Get instance
     *
     * @return Widgets_Loader
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    public function __construct() {
        // Register hooks for future calls
        add_action('elementor/widgets/register', array($this, 'register_widgets'));
        add_action('elementor/elements/categories_registered', array($this, 'register_categories'));
        add_action('elementor/frontend/after_enqueue_styles', array($this, 'enqueue_widget_styles'));
        add_action('elementor/editor/after_enqueue_styles', array($this, 'enqueue_editor_styles'));

        // If elementor/widgets/register already fired, register immediately
        if (did_action('elementor/widgets/register') && class_exists('\Elementor\Plugin')) {
            $this->register_widgets(\Elementor\Plugin::instance()->widgets_manager);
        }

        // If elementor/elements/categories_registered already fired, register categories immediately
        if (did_action('elementor/elements/categories_registered') && class_exists('\Elementor\Plugin')) {
            $this->register_categories(\Elementor\Plugin::instance()->elements_manager);
        }
    }

    /**
     * Register widget categories
     */
    public function register_categories($elements_manager) {
        $categories = array(
            'ai-theme-navbar' => __('AI Theme - Navbar', 'ai-theme'),
            'ai-theme-hero' => __('AI Theme - Hero', 'ai-theme'),
            'ai-theme-about' => __('AI Theme - About', 'ai-theme'),
            'ai-theme-services' => __('AI Theme - Services', 'ai-theme'),
            'ai-theme-features' => __('AI Theme - Features', 'ai-theme'),
            'ai-theme-contact' => __('AI Theme - Contact', 'ai-theme'),
            'ai-theme-cta' => __('AI Theme - CTA', 'ai-theme'),
            'ai-theme-team' => __('AI Theme - Team', 'ai-theme'),
            'ai-theme-testimonials' => __('AI Theme - Testimonials', 'ai-theme'),
            'ai-theme-faq' => __('AI Theme - FAQ', 'ai-theme'),
            'ai-theme-pricing' => __('AI Theme - Pricing', 'ai-theme'),
            'ai-theme-stats' => __('AI Theme - Stats', 'ai-theme'),
            'ai-theme-gallery' => __('AI Theme - Gallery', 'ai-theme'),
            'ai-theme-video' => __('AI Theme - Video', 'ai-theme'),
            'ai-theme-timeline' => __('AI Theme - Timeline', 'ai-theme'),
            'ai-theme-portfolio' => __('AI Theme - Portfolio', 'ai-theme'),
            'ai-theme-cases' => __('AI Theme - Cases', 'ai-theme'),
            'ai-theme-company' => __('AI Theme - Company', 'ai-theme'),
            'ai-theme-comparison' => __('AI Theme - Comparison', 'ai-theme'),
            'ai-theme-integrations' => __('AI Theme - Integrations', 'ai-theme'),
            'ai-theme-map' => __('AI Theme - Map', 'ai-theme'),
            'ai-theme-newsletter' => __('AI Theme - Newsletter', 'ai-theme'),
            'ai-theme-blog' => __('AI Theme - Blog', 'ai-theme'),
            'ai-theme-shop' => __('AI Theme - Shop', 'ai-theme'),
            'ai-theme-cart' => __('AI Theme - Cart', 'ai-theme'),
            'ai-theme-categories' => __('AI Theme - Categories', 'ai-theme'),
            'ai-theme-filters' => __('AI Theme - Filters', 'ai-theme'),
            'ai-theme-product' => __('AI Theme - Product', 'ai-theme'),
            'ai-theme-products' => __('AI Theme - Products', 'ai-theme'),
            'ai-theme-reviews' => __('AI Theme - Reviews', 'ai-theme'),
            'ai-theme-shipping' => __('AI Theme - Shipping', 'ai-theme'),
            'ai-theme-modal' => __('AI Theme - Modal', 'ai-theme'),
            'ai-theme-text' => __('AI Theme - Text', 'ai-theme'),
            'ai-theme-process' => __('AI Theme - Process', 'ai-theme'),
            'ai-theme-breadcrumbs' => __('AI Theme - Breadcrumbs', 'ai-theme'),
            'ai-theme-footer' => __('AI Theme - Footer', 'ai-theme'),
            'ai-theme-privacy' => __('AI Theme - Privacy', 'ai-theme'),
            'ai-theme-cookies' => __('AI Theme - Cookies', 'ai-theme'),
            'ai-theme-legal' => __('AI Theme - Legal', 'ai-theme'),
            'ai-theme-general' => __('AI Theme - General', 'ai-theme'),
        );

        foreach ($categories as $slug => $title) {
            $elements_manager->add_category($slug, ['title' => $title, 'icon' => 'fa fa-plug']);
        }
    }

    /**
     * Register widgets
     *
     * @param \Elementor\Widgets_Manager $widgets_manager
     */
    public function register_widgets($widgets_manager) {
        // Include widget files
        $this->include_widgets();

        // Register widgets
        $widgets = $this->get_widgets();

        foreach ($widgets as $widget_class) {
            if (class_exists($widget_class)) {
                $widgets_manager->register(new $widget_class());
            }
        }
    }

    /**
     * Include widget files
     */
    private function include_widgets() {
        $elementor_dir = AI_THEME_DIR . '/inc/elementor/';
        $widgets_dir = $elementor_dir . 'widgets/';

        // Base widget class first (required for all widgets)
        $base_class_file = $elementor_dir . 'class-ai-widget-base.php';
        if (file_exists($base_class_file)) {
            require_once $base_class_file;
        }

        // Auto-load all widget files
        $widget_files = glob($widgets_dir . 'class-*.php');
        foreach ($widget_files as $file) {
            require_once $file;
        }
    }

    /**
     * Get registered widgets - automatically discover all widget classes from files
     *
     * @return array
     */
    private function get_widgets() {
        $widgets = array();
        $widgets_dir = AI_THEME_DIR . '/inc/elementor/widgets/';
        
        // Get all widget files
        $widget_files = glob($widgets_dir . 'class-*.php');
        
        foreach ($widget_files as $file) {
            // Extract class name from filename: class-hero-minimal.php -> Hero_Minimal
            $filename = basename($file, '.php');
            $class_suffix = str_replace('class-', '', $filename);
            $class_suffix = str_replace('-', '_', $class_suffix);
            $class_suffix = implode('_', array_map('ucfirst', explode('_', $class_suffix)));
            
            $widget_class = 'AITheme\\Elementor\\Widgets\\' . $class_suffix;
            $widgets[] = $widget_class;
        }
        
        return $widgets;
    }

    /**
     * Enqueue widget styles for frontend
     */
    public function enqueue_widget_styles() {
        // Styles are already loaded via theme
    }

    /**
     * Enqueue styles for Elementor editor
     */
    public function enqueue_editor_styles() {
        wp_enqueue_style(
            'ai-theme-editor-widgets',
            AI_THEME_URI . '/assets/css/utilities.css',
            array(),
            AI_THEME_VERSION
        );

        wp_enqueue_style(
            'ai-theme-editor-themes',
            AI_THEME_URI . '/assets/css/themes.css',
            array(),
            AI_THEME_VERSION
        );

        wp_enqueue_style(
            'ai-theme-editor-blocks',
            AI_THEME_URI . '/assets/css/block-themes.css',
            array(),
            AI_THEME_VERSION
        );
    }
}

// Initialize when loaded - either via elementor/init or directly
if (did_action('elementor/init')) {
    // elementor/init already fired, initialize immediately
    Widgets_Loader::instance();
} else {
    // Wait for elementor/init
    add_action('elementor/init', function() {
        Widgets_Loader::instance();
    });
}
