<?php
namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

class Blog_Featured extends AI_Widget_Base {
    public function get_name() { return 'ai_blog_featured'; }
    public function get_title() { return __('Blog Featured (WordPress Posts)', 'ai-theme'); }
    public function get_icon() { return 'eicon-featured-image'; }
    public function get_categories() { return array('ai-theme-blog'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', ['label' => __('Content', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT]);

        $this->add_control('title', [
            'label' => __('Title', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ]);

        $this->add_control('featured_count', [
            'label' => __('Number of Featured Posts', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 1,
            'min' => 1,
            'max' => 3,
        ]);

        $this->add_control('regular_count', [
            'label' => __('Number of Regular Posts', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 4,
            'min' => 0,
            'max' => 12,
        ]);

        $this->add_control('use_sticky', [
            'label' => __('Use Sticky Posts as Featured', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $categories = $this->get_post_categories();
        $this->add_control('category', [
            'label' => __('Category', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => '',
            'options' => $categories,
        ]);

        $this->end_controls_section();
    }

    private function get_post_categories() {
        $categories = ['' => __('All Categories', 'ai-theme')];
        $terms = get_categories(['hide_empty' => true]);
        if (!empty($terms) && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[$term->slug] = $term->name;
            }
        }
        return $categories;
    }

    protected function render() {
        $s = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $featured = $this->get_featured_posts($s);
        $regular = $this->get_regular_posts($s, wp_list_pluck($featured, 'ID'));
        ?>
        <section class="ai-blog ai-blog-featured section-secondary py-16" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="blog" data-block-variant="blog-featured">
            <div class="ai-container">
                <div class="text-center mb-12">
                    <?php if (!empty($s['title'])) : ?><h2 class="text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($s['title']); ?></h2><?php endif; ?>
                </div>

                <?php if (!empty($featured)) : ?>
                <div class="grid md:grid-cols-2 gap-8 mb-8">
                    <?php foreach ($featured as $post) : $this->render_featured_post($post); endforeach; ?>
                </div>
                <?php endif; ?>

                <?php if (!empty($regular)) : ?>
                <div class="ai-blog-posts grid md:grid-cols-2 gap-8">
                    <?php foreach ($regular as $post) : $this->render_regular_post($post); endforeach; ?>
                </div>
                <?php endif; ?>

                <?php if (empty($featured) && empty($regular)) : ?>
                <div class="text-center py-12">
                    <p class="text-[var(--text-muted)]"><?php echo esc_html(ai_get_text('blocks.blog.noPosts', '')); ?></p>
                </div>
                <?php endif; ?>
            </div>
        </section>
        <?php
    }

    private function get_featured_posts($settings) {
        $args = [
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $settings['featured_count'] ?? 1,
            'orderby' => 'date',
            'order' => 'DESC',
        ];

        if ($settings['use_sticky'] === 'yes') {
            $sticky = get_option('sticky_posts');
            if (!empty($sticky)) {
                $args['post__in'] = $sticky;
                $args['ignore_sticky_posts'] = 1;
            }
        }

        if (!empty($settings['category'])) {
            $args['category_name'] = $settings['category'];
        }

        return (new \WP_Query($args))->posts;
    }

    private function get_regular_posts($settings, $exclude_ids = []) {
        if (($settings['regular_count'] ?? 4) < 1) return [];

        $args = [
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $settings['regular_count'] ?? 4,
            'orderby' => 'date',
            'order' => 'DESC',
            'post__not_in' => $exclude_ids,
            'ignore_sticky_posts' => 1,
        ];

        if (!empty($settings['category'])) {
            $args['category_name'] = $settings['category'];
        }

        return (new \WP_Query($args))->posts;
    }

    private function render_featured_post($post) {
        $thumbnail_url = get_the_post_thumbnail_url($post->ID, 'large');
        $categories = get_the_category($post->ID);
        $primary_category = !empty($categories) ? $categories[0] : null;
        ?>
        <article class="ai-blog-post ai-blog-post-featured md:col-span-2 bg-[var(--bg-primary)] rounded-3xl overflow-hidden shadow-lg hover:shadow-2xl transition-all">
            <div class="grid md:grid-cols-2">
                <?php if ($thumbnail_url) : ?>
                <a href="<?php echo esc_url(get_permalink($post)); ?>">
                    <img src="<?php echo esc_url($thumbnail_url); ?>" alt="<?php echo esc_attr($post->post_title); ?>" class="w-full h-64 md:h-full object-cover">
                </a>
                <?php endif; ?>
                <div class="ai-blog-content p-8 flex flex-col justify-center">
                    <?php if ($primary_category) : ?>
                    <a href="<?php echo esc_url(get_category_link($primary_category->term_id)); ?>" class="ai-blog-category inline-block px-3 py-1 bg-[var(--primary)] text-white text-sm font-semibold rounded-full mb-4 self-start hover:opacity-90 transition-colors">
                        <?php echo esc_html($primary_category->name); ?>
                    </a>
                    <?php endif; ?>
                    <h3 class="text-3xl font-bold text-[var(--text-primary)] mb-4">
                        <a href="<?php echo esc_url(get_permalink($post)); ?>"><?php echo esc_html($post->post_title); ?></a>
                    </h3>
                    <p class="text-[var(--text-secondary)] mb-6"><?php echo esc_html(wp_trim_words(get_the_excerpt($post), 30)); ?></p>
                    <span class="ai-blog-date text-sm text-[var(--text-muted)] mb-4"><?php echo esc_html(get_the_date('', $post)); ?></span>
                    <a href="<?php echo esc_url(get_permalink($post)); ?>" class="self-start bg-[var(--primary)] text-white px-6 py-3 rounded-lg font-medium hover:opacity-90 transition-colors">
                        <?php echo esc_html(ai_get_text('blocks.blog.readMore', '')); ?> →
                    </a>
                </div>
            </div>
        </article>
        <?php
    }

    private function render_regular_post($post) {
        $thumbnail_url = get_the_post_thumbnail_url($post->ID, 'medium');
        ?>
        <article class="ai-blog-post bg-[var(--bg-primary)] rounded-2xl overflow-hidden shadow-sm hover:shadow-lg transition-all">
            <?php if ($thumbnail_url) : ?>
            <a href="<?php echo esc_url(get_permalink($post)); ?>">
                <img src="<?php echo esc_url($thumbnail_url); ?>" alt="<?php echo esc_attr($post->post_title); ?>" class="w-full h-48 object-cover">
            </a>
            <?php endif; ?>
            <div class="p-6">
                <h3 class="text-xl font-bold text-[var(--text-primary)] mb-2">
                    <a href="<?php echo esc_url(get_permalink($post)); ?>"><?php echo esc_html($post->post_title); ?></a>
                </h3>
                <span class="ai-blog-date text-sm text-[var(--text-muted)]"><?php echo esc_html(get_the_date('', $post)); ?></span>
            </div>
        </article>
        <?php
    }
}
