<?php
namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

class Blog_Sidebar extends AI_Widget_Base {
    public function get_name() { return 'ai_blog_sidebar'; }
    public function get_title() { return __('Blog Magazine', 'ai-theme'); }
    public function get_icon() { return 'eicon-posts-justified'; }
    public function get_categories() { return array('ai-theme-blog'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', ['label' => __('Content', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT]);

        $this->add_control('title', [
            'label' => __('Title', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ]);

        $this->add_control('subtitle', [
            'label' => __('Subtitle', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ]);

        $this->add_control('posts_per_page', [
            'label' => __('Number of Posts', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 4,
            'min' => 1,
            'max' => 10,
        ]);

        $this->add_control('orderby', [
            'label' => __('Order By', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => 'date',
            'options' => [
                'date' => __('Date', 'ai-theme'),
                'title' => __('Title', 'ai-theme'),
                'modified' => __('Last Modified', 'ai-theme'),
                'rand' => __('Random', 'ai-theme'),
            ],
        ]);

        $this->end_controls_section();
    }

    protected function render() {
        $s = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $posts = $this->get_posts($s);
        
        if (empty($posts)) {
            return;
        }

        $featured = $posts[0];
        $other_posts = array_slice($posts, 1, 3);
        ?>
        <section class="ai-blog ai-blog-sidebar section-light py-16" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="blog" data-block-variant="blog-sidebar">
            <div class="max-w-7xl mx-auto px-6">
                <?php if (!empty($s['title']) || !empty($s['subtitle'])) : ?>
                <div class="text-center mb-12">
                    <?php if (!empty($s['title'])) : ?>
                    <h2 class="text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($s['title']); ?></h2>
                    <?php endif; ?>
                    <?php if (!empty($s['subtitle'])) : ?>
                    <p class="text-lg text-[var(--text-secondary)] max-w-2xl mx-auto"><?php echo esc_html($s['subtitle']); ?></p>
                    <?php endif; ?>
                </div>
                <?php endif; ?>

                <!-- Featured Post -->
                <article class="mb-12">
                    <a href="<?php echo esc_url(get_permalink($featured)); ?>" class="group block">
                        <div class="relative overflow-hidden rounded-2xl">
                            <?php 
                            $featured_img = get_the_post_thumbnail_url($featured->ID, 'large');
                            if (!$featured_img) $featured_img = 'https://placehold.co/1200x600/e2e8f0/475569?text=Featured';
                            ?>
                            <img src="<?php echo esc_url($featured_img); ?>" alt="<?php echo esc_attr($featured->post_title); ?>" class="w-full h-[400px] object-cover group-hover:scale-105 transition-transform duration-500">
                            <div class="absolute inset-0 bg-gradient-to-t from-black/80 via-black/40 to-transparent"></div>
                            <div class="absolute bottom-0 left-0 right-0 p-8">
                                <?php 
                                $categories = get_the_category($featured->ID);
                                if (!empty($categories)) :
                                    $cat_color = $this->get_category_color($categories[0]->name);
                                ?>
                                <span class="inline-block px-3 py-1 <?php echo esc_attr($cat_color); ?> text-white text-sm font-medium rounded-full mb-4">
                                    <?php echo esc_html($categories[0]->name); ?>
                                </span>
                                <?php endif; ?>
                                <h3 class="text-3xl font-bold text-white mb-3 group-hover:text-[var(--primary-light,#93c5fd)] transition-colors">
                                    <?php echo esc_html($featured->post_title); ?>
                                </h3>
                                <p class="text-white/80 text-lg mb-4 line-clamp-2">
                                    <?php echo esc_html(wp_trim_words(get_the_excerpt($featured), 25)); ?>
                                </p>
                                <div class="flex items-center gap-4 text-sm text-white/70">
                                    <span class="font-medium"><?php echo esc_html(get_the_author_meta('display_name', $featured->post_author)); ?></span>
                                    <span>•</span>
                                    <span><?php echo esc_html(get_the_date('', $featured)); ?></span>
                                </div>
                            </div>
                        </div>
                    </a>
                </article>

                <?php if (!empty($other_posts)) : ?>
                <!-- Other Posts Grid -->
                <div class="grid md:grid-cols-3 gap-8">
                    <?php foreach ($other_posts as $post) : 
                        $thumb = get_the_post_thumbnail_url($post->ID, 'medium_large');
                        if (!$thumb) $thumb = 'https://placehold.co/400x300/e2e8f0/475569?text=Blog';
                        $cats = get_the_category($post->ID);
                    ?>
                    <article class="group">
                        <a href="<?php echo esc_url(get_permalink($post)); ?>" class="block">
                            <div class="relative overflow-hidden rounded-xl mb-4">
                                <img src="<?php echo esc_url($thumb); ?>" alt="<?php echo esc_attr($post->post_title); ?>" class="w-full h-48 object-cover group-hover:scale-105 transition-transform duration-300">
                                <?php if (!empty($cats)) : ?>
                                <span class="absolute top-3 left-3 px-2 py-1 <?php echo esc_attr($this->get_category_color($cats[0]->name)); ?> text-white text-xs font-medium rounded-full">
                                    <?php echo esc_html($cats[0]->name); ?>
                                </span>
                                <?php endif; ?>
                            </div>
                            <div class="flex items-center gap-2 text-sm text-[var(--text-muted)] mb-2">
                                <span><?php echo esc_html(get_the_date('', $post)); ?></span>
                            </div>
                            <h3 class="text-xl font-bold text-[var(--text-primary)] mb-2 group-hover:text-[var(--primary)] transition-colors line-clamp-2">
                                <?php echo esc_html($post->post_title); ?>
                            </h3>
                            <p class="text-[var(--text-secondary)] text-sm line-clamp-2">
                                <?php echo esc_html(wp_trim_words(get_the_excerpt($post), 15)); ?>
                            </p>
                        </a>
                    </article>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </section>
        <?php
    }

    private function get_posts($settings) {
        $args = [
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $settings['posts_per_page'] ?? 4,
            'orderby' => $settings['orderby'] ?? 'date',
            'order' => 'DESC',
        ];

        return (new \WP_Query($args))->posts;
    }

    private function get_category_color($category_name) {
        $hash = crc32($category_name);
        $colors = ['bg-blue-500', 'bg-purple-500', 'bg-green-500', 'bg-orange-500', 'bg-pink-500', 'bg-indigo-500'];
        return $colors[$hash % count($colors)];
    }
}
