<?php
/**
 * Cart Cards Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cart Cards Widget - Card-based cart display
 */
class Cart_Cards extends AI_Widget_Base {

    public function get_name() {
        return 'ai_cart_cards';
    }

    public function get_title() {
        return __('Cart Cards', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-cart';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'title',
            array(
                'label'   => __('Title', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'empty_text',
            array(
                'label'   => __('Empty Cart Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'checkout_text',
            array(
                'label'   => __('Checkout Button Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'continue_text',
            array(
                'label'   => __('Continue Shopping Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        // Check if WooCommerce cart is initialized (may be null during REST API requests)
        if (!WC() || !WC()->cart) {
            return;
        }

        $cart = WC()->cart;
        $cart_items = $cart->get_cart();
        ?>
        <section 
            class="py-12 bg-[var(--bg-secondary)]"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="cart"
            data-block-variant="cart-cards"
        >
            <div class="max-w-6xl mx-auto px-4">
                <div class="flex items-center gap-3 mb-8">
                    <svg class="w-8 h-8 text-[var(--primary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                    <h1 class="text-2xl font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['title']); ?></h1>
                </div>

                <?php if (empty($cart_items)) : ?>
                    <div class="max-w-md mx-auto text-center py-16">
                        <div class="w-20 h-20 bg-[var(--bg-primary)] rounded-full flex items-center justify-center mx-auto mb-6">
                            <svg class="w-10 h-10 text-[var(--text-muted)]" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                        </div>
                        <h2 class="text-2xl font-bold text-[var(--text-primary)] mb-2"><?php echo esc_html($settings['empty_text']); ?></h2>
                        <a href="<?php echo esc_url(wc_get_page_permalink('shop')); ?>" class="inline-block mt-4 px-6 py-3 bg-[var(--primary)] text-white rounded-[var(--btn-radius)] font-semibold hover:opacity-90">
                            <?php echo esc_html($settings['continue_text']); ?>
                        </a>
                    </div>
                <?php else : ?>
                    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                        <!-- Product Cards -->
                        <div class="lg:col-span-2 grid grid-cols-1 sm:grid-cols-2 gap-4">
                            <?php foreach ($cart_items as $cart_item_key => $cart_item) : 
                                $product = $cart_item['data'];
                                $product_id = $cart_item['product_id'];
                                $quantity = $cart_item['quantity'];
                                $image = wp_get_attachment_image_url($product->get_image_id(), 'woocommerce_thumbnail');
                            ?>
                                <div class="bg-[var(--bg-primary)] rounded-[var(--card-radius)] shadow-[var(--shadow-card)] overflow-hidden">
                                    <?php if ($image) : ?>
                                        <div class="aspect-square bg-[var(--bg-secondary)]">
                                            <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover" />
                                        </div>
                                    <?php endif; ?>
                                    <div class="p-4">
                                        <h3 class="font-semibold text-[var(--text-primary)] mb-1 truncate"><?php echo esc_html($product->get_name()); ?></h3>
                                        
                                        <div class="flex items-center justify-between mt-3">
                                            <div class="flex items-center gap-2 bg-[var(--bg-secondary)] rounded-full p-1">
                                                <span class="px-3 font-medium"><?php echo esc_html($quantity); ?></span>
                                            </div>
                                            <p class="text-lg font-bold text-[var(--text-primary)]">
                                                <?php echo wc_price($product->get_price() * $quantity); ?>
                                            </p>
                                        </div>

                                        <a href="<?php echo esc_url(wc_get_cart_remove_url($cart_item_key)); ?>" class="mt-3 w-full py-2 text-sm text-red-500 hover:bg-red-50 rounded-[var(--btn-radius)] flex items-center justify-center gap-1">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
                                            <?php _e('Remove', 'ai-theme'); ?>
                                        </a>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>

                        <!-- Summary Card -->
                        <div class="lg:col-span-1">
                            <div class="bg-[var(--bg-primary)] rounded-[var(--card-radius)] shadow-[var(--shadow-card)] p-6 sticky top-4">
                                <h2 class="text-lg font-bold text-[var(--text-primary)] mb-6"><?php _e('Order Summary', 'ai-theme'); ?></h2>
                                
                                <div class="space-y-3 mb-6">
                                    <div class="flex justify-between text-[var(--text-secondary)]">
                                        <span><?php _e('Subtotal', 'ai-theme'); ?></span>
                                        <span><?php echo wc_price($cart->get_subtotal()); ?></span>
                                    </div>
                                    <?php if ($cart->get_shipping_total() > 0) : ?>
                                        <div class="flex justify-between text-[var(--text-secondary)]">
                                            <span><?php _e('Shipping', 'ai-theme'); ?></span>
                                            <span><?php echo wc_price($cart->get_shipping_total()); ?></span>
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <div class="flex justify-between py-4 border-t border-[var(--border-color)]">
                                    <span class="text-lg font-semibold text-[var(--text-primary)]"><?php _e('Total', 'ai-theme'); ?></span>
                                    <span class="text-2xl font-bold text-[var(--primary)]"><?php echo wc_price($cart->get_total('')); ?></span>
                                </div>

                                <a href="<?php echo esc_url(wc_get_checkout_url()); ?>" class="block w-full mt-4 bg-[var(--primary)] text-white py-4 text-center rounded-[var(--btn-radius)] font-semibold hover:opacity-90 transition-opacity">
                                    <?php echo esc_html($settings['checkout_text']); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </section>
        <?php
    }
}
