<?php
/**
 * Navbar Centered Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

class Navbar_Centered extends AI_Widget_Base {

    public function get_name() {
        return 'ai_navbar_centered';
    }

    public function get_title() {
        return __('Navbar Centered', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-nav-menu';
    }

    public function get_categories() {
        return array('ai-theme-navigation');
    }

    public function get_keywords() {
        return array('navbar', 'navigation', 'menu', 'header', 'centered', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'logo_text',
            array(
                'label'   => __('Logo Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'logo_image',
            array(
                'label' => __('Logo Image', 'ai-theme'),
                'type'  => Controls_Manager::MEDIA,
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'title',
            array(
                'label'   => __('Title', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'link',
            array(
                'label' => __('Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $this->add_control(
            'menu_items',
            array(
                'label'   => __('Menu Items', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(
                    array('title' => 'Home', 'link' => array('url' => '/')),
                    array('title' => 'About', 'link' => array('url' => '/about')),
                    array('title' => 'Contact', 'link' => array('url' => '/contact')),
                ),
            )
        );

        $this->add_control(
            'show_cart',
            array(
                'label'        => __('Show Cart', 'ai-theme'),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __('Yes', 'ai-theme'),
                'label_off'    => __('No', 'ai-theme'),
                'return_value' => 'yes',
                'default'      => '',
            )
        );

        $this->add_control(
            'cta_text',
            array(
                'label'   => __('CTA Button Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
                'description' => __('CTA button for landing pages (shown when cart is hidden)', 'ai-theme'),
            )
        );

        $this->add_control(
            'cta_link',
            array(
                'label' => __('CTA Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
                'default' => array('url' => '#contact'),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $unique_id = 'navbar-centered-' . $this->get_id();

        // WooCommerce
        $cart_url = function_exists('wc_get_cart_url') ? wc_get_cart_url() : home_url('/cart/');
        $cart_count = 0;
        if (function_exists('WC') && WC()->cart) {
            $cart_count = WC()->cart->get_cart_contents_count();
        }
        ?>
        <nav class="ai-navbar ai-navbar-centered bg-[var(--bg-primary)] border-b border-[var(--border-color)] sticky top-0 z-50" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="navbar" data-block-variant="navbar-centered" id="<?php echo esc_attr($unique_id); ?>">
            <div class="ai-container py-4">
                <!-- Mobile: Logo left, hamburger right -->
                <div class="flex md:hidden items-center justify-between">
                    <a href="<?php echo esc_url(home_url('/')); ?>" class="flex items-center gap-2">
                        <?php if (!empty($settings['logo_image']['url'])) : ?>
                            <img src="<?php echo esc_url($settings['logo_image']['url']); ?>" alt="Logo" class="h-8 object-contain">
                        <?php endif; ?>
                        <?php if (empty($settings['logo_image']['url']) && !empty($settings['logo_text'])) : ?>
                            <div class="w-8 h-8 bg-gradient-to-br from-[var(--primary)] to-[var(--secondary)] rounded-lg flex items-center justify-center text-white font-bold">
                                <?php echo esc_html(mb_substr($settings['logo_text'], 0, 1)); ?>
                            </div>
                        <?php endif; ?>
                        <?php if (!empty($settings['logo_text'])) : ?>
                            <span class="text-xl font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['logo_text']); ?></span>
                        <?php endif; ?>
                    </a>

                    <div class="flex items-center gap-2">
                        <?php if ($settings['show_cart'] === 'yes') : ?>
                            <a href="<?php echo esc_url($cart_url); ?>" class="p-2 hover:bg-[var(--bg-secondary)] rounded-lg transition-colors relative">
                                <svg class="w-5 h-5 text-[var(--text-secondary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                </svg>
                                <?php if ($cart_count > 0) : ?>
                                <span class="absolute -top-1 -right-1 bg-[var(--primary)] text-white text-xs w-5 h-5 rounded-full flex items-center justify-center"><?php echo esc_html($cart_count); ?></span>
                                <?php endif; ?>
                            </a>
                        <?php endif; ?>

                        <button class="p-2 hover:bg-[var(--bg-secondary)] rounded-lg ai-mobile-menu-toggle" aria-label="Toggle menu">
                            <svg class="w-6 h-6 ai-menu-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                            </svg>
                            <svg class="w-6 h-6 ai-close-icon hidden" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                        </button>
                    </div>
                </div>

                <!-- Desktop: Centered logo and menu -->
                <div class="hidden md:block">
                    <div class="text-center mb-4">
                        <a href="<?php echo esc_url(home_url('/')); ?>" class="inline-flex items-center gap-2">
                            <?php if (!empty($settings['logo_image']['url'])) : ?>
                                <img src="<?php echo esc_url($settings['logo_image']['url']); ?>" alt="Logo" class="h-10 object-contain">
                            <?php endif; ?>
                            <?php if (empty($settings['logo_image']['url']) && !empty($settings['logo_text'])) : ?>
                                <div class="w-10 h-10 bg-gradient-to-br from-[var(--primary)] to-[var(--secondary)] rounded-lg flex items-center justify-center text-white font-bold text-lg">
                                    <?php echo esc_html(mb_substr($settings['logo_text'], 0, 1)); ?>
                                </div>
                            <?php endif; ?>
                            <?php if (!empty($settings['logo_text'])) : ?>
                                <span class="text-2xl font-bold text-[var(--text-primary)]"><?php echo esc_html($settings['logo_text']); ?></span>
                            <?php endif; ?>
                        </a>
                    </div>
                    <div class="flex items-center justify-center gap-8">
                        <?php foreach ($settings['menu_items'] as $item) : ?>
                            <a
                                href="<?php echo esc_url($item['link']['url'] ?? '#'); ?>"
                                class="text-[var(--text-secondary)] hover:text-[var(--primary)] transition-colors font-medium"
                            >
                                <?php echo esc_html($item['title']); ?>
                            </a>
                        <?php endforeach; ?>

                        <?php if ($settings['show_cart'] === 'yes') : ?>
                            <a href="<?php echo esc_url($cart_url); ?>" class="p-2 hover:bg-[var(--bg-secondary)] rounded-lg transition-colors relative">
                                <svg class="w-5 h-5 text-[var(--text-secondary)]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                </svg>
                                <?php if ($cart_count > 0) : ?>
                                <span class="absolute -top-1 -right-1 bg-[var(--primary)] text-white text-xs w-5 h-5 rounded-full flex items-center justify-center"><?php echo esc_html($cart_count); ?></span>
                                <?php endif; ?>
                            </a>
                        <?php endif; ?>

                        <?php
                        // CTA Button - показывается для лендингов (когда cart скрыт)
                        if (!empty($settings['cta_text']) && $settings['show_cart'] !== 'yes') :
                            $cta_url = !empty($settings['cta_link']['url']) ? $settings['cta_link']['url'] : '#contact';
                        ?>
                            <a href="<?php echo esc_url($cta_url); ?>" class="px-4 py-2 bg-[var(--primary)] text-white rounded-lg hover:opacity-90 transition-opacity font-medium">
                                <?php echo esc_html($settings['cta_text']); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Mobile Menu -->
                <div class="ai-mobile-menu md:hidden hidden border-t border-[var(--border-color)] mt-4">
                    <div class="py-4 space-y-2">
                        <?php foreach ($settings['menu_items'] as $item) : ?>
                            <a
                                href="<?php echo esc_url($item['link']['url'] ?? '#'); ?>"
                                class="block px-4 py-2 text-[var(--text-secondary)] hover:bg-[var(--bg-secondary)] hover:text-[var(--primary)] rounded-lg transition-colors font-medium"
                            >
                                <?php echo esc_html($item['title']); ?>
                            </a>
                        <?php endforeach; ?>

                        <?php
                        // CTA Button в мобильном меню
                        if (!empty($settings['cta_text']) && $settings['show_cart'] !== 'yes') :
                            $cta_url = !empty($settings['cta_link']['url']) ? $settings['cta_link']['url'] : '#contact';
                        ?>
                        <a href="<?php echo esc_url($cta_url); ?>" class="block mx-4 mt-2 px-4 py-2 bg-[var(--primary)] text-white text-center rounded-lg font-medium">
                            <?php echo esc_html($settings['cta_text']); ?>
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </nav>

        <script>
        document.addEventListener('DOMContentLoaded', function() {
            var navbar = document.getElementById('<?php echo esc_js($unique_id); ?>');
            if (navbar) {
                var toggle = navbar.querySelector('.ai-mobile-menu-toggle');
                var menu = navbar.querySelector('.ai-mobile-menu');
                var menuIcon = navbar.querySelector('.ai-menu-icon');
                var closeIcon = navbar.querySelector('.ai-close-icon');

                if (toggle && menu) {
                    toggle.addEventListener('click', function() {
                        menu.classList.toggle('hidden');
                        menuIcon.classList.toggle('hidden');
                        closeIcon.classList.toggle('hidden');
                    });
                }
            }
        });
        </script>
        <?php
    }
}
