<?php
namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) exit;

class Privacy_Default extends AI_Widget_Base {
    public function get_name() { return 'ai_privacy_default'; }
    public function get_title() { return __('Privacy Policy', 'ai-theme'); }
    public function get_icon() { return 'eicon-document-file'; }
    public function get_categories() { return array('ai-theme-privacy'); }

    protected function register_controls() {
        // Content Section
        $this->start_controls_section('content_section', [
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT
        ]);

        $this->add_control('language', [
            'label' => __('Language', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'options' => ['ru' => 'Russian', 'en' => 'English'],
            'default' => 'ru'
        ]);

        $this->add_control('title', [
            'label' => __('Title', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => ''
        ]);

        $this->add_control('last_updated', [
            'label' => __('Last Updated', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => ''
        ]);

        $this->end_controls_section();

        // Company Details Section
        $this->start_controls_section('company_section', [
            'label' => __('Company Details', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT
        ]);

        $this->add_control('company_name', [
            'label' => __('Company Name', 'ai-theme'),
            'type' => Controls_Manager::TEXT
        ]);

        $this->add_control('legal_name', [
            'label' => __('Legal Name', 'ai-theme'),
            'type' => Controls_Manager::TEXT
        ]);

        $this->add_control('contact_email', [
            'label' => __('Contact Email', 'ai-theme'),
            'type' => Controls_Manager::TEXT
        ]);

        $this->add_control('phone', [
            'label' => __('Phone', 'ai-theme'),
            'type' => Controls_Manager::TEXT
        ]);

        $this->add_control('address', [
            'label' => __('Address', 'ai-theme'),
            'type' => Controls_Manager::TEXTAREA,
            'rows' => 2
        ]);

        $this->add_control('inn', [
            'label' => __('INN / Tax ID', 'ai-theme'),
            'type' => Controls_Manager::TEXT
        ]);

        $this->add_control('ogrn', [
            'label' => __('OGRN / Registration Number', 'ai-theme'),
            'type' => Controls_Manager::TEXT
        ]);

        $this->add_control('director', [
            'label' => __('Director', 'ai-theme'),
            'type' => Controls_Manager::TEXT
        ]);

        $this->end_controls_section();

        // Custom Sections
        $this->start_controls_section('sections_section', [
            'label' => __('Custom Sections', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT
        ]);

        $this->add_control('use_custom_sections', [
            'label' => __('Use Custom Sections', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => ''
        ]);

        $repeater = new Repeater();
        $repeater->add_control('section_title', [
            'label' => __('Section Title', 'ai-theme'),
            'type' => Controls_Manager::TEXT
        ]);
        $repeater->add_control('section_content', [
            'label' => __('Section Content', 'ai-theme'),
            'type' => Controls_Manager::TEXTAREA,
            'rows' => 4
        ]);

        $this->add_control('sections', [
            'label' => __('Sections', 'ai-theme'),
            'type' => Controls_Manager::REPEATER,
            'fields' => $repeater->get_controls(),
            'title_field' => '{{{ section_title }}}'
        ]);

        $this->end_controls_section();
    }

    protected function render() {
        $s = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        // Use AI-generated texts with fallbacks
        $t = [
            'title' => ai_get_text('legal.privacy.title', 'Privacy Policy'),
            'lastUpdated' => ai_get_text('legal.privacy.lastUpdated', 'Last updated'),
            'operatorTitle' => ai_get_text('legal.privacy.dataController', 'Data Controller'),
            'operatorText' => ai_get_text('legal.privacy.dataControllerText', 'The data controller is:'),
            'contactTitle' => ai_get_text('legal.privacy.contactTitle', 'Contact Information'),
            'contactText' => ai_get_text('legal.privacy.contactText', 'If you have any questions about this Privacy Policy or the processing of your personal data, please contact us:'),
            'addressLabel' => ai_get_text('common.form.address', 'Address'),
            'phoneLabel' => ai_get_text('common.form.phone', 'Phone'),
            'emailLabel' => ai_get_text('common.form.email', 'Email'),
            'innLabel' => ai_get_text('legal.privacy.taxId', 'Tax ID'),
            'ogrnLabel' => ai_get_text('legal.privacy.registrationNumber', 'Registration Number'),
            'directorLabel' => ai_get_text('legal.privacy.director', 'Director'),
        ];

        // Default sections - these should come from widget settings or AI-generated content
        $defaultSections = [
            ['title' => ai_get_text('legal.privacy.sections.collection.title', 'Information Collection'), 'content' => ai_get_text('legal.privacy.sections.collection.content', 'We collect information that you provide to us directly, such as when creating an account, filling out a contact form, or subscribing to our newsletter.')],
            ['title' => ai_get_text('legal.privacy.sections.usage.title', 'Use of Information'), 'content' => ai_get_text('legal.privacy.sections.usage.content', 'The collected information is used to provide and improve our services, process your requests, send notifications and marketing materials.')],
            ['title' => ai_get_text('legal.privacy.sections.cookies.title', 'Cookies and Tracking Technologies'), 'content' => ai_get_text('legal.privacy.sections.cookies.content', 'Our website uses cookies and similar technologies to improve user experience, analyze traffic, and personalize content.')],
            ['title' => ai_get_text('legal.privacy.sections.sharing.title', 'Sharing Data with Third Parties'), 'content' => ai_get_text('legal.privacy.sections.sharing.content', 'We do not sell or share your personal information with third parties, except as required by law.')],
            ['title' => ai_get_text('legal.privacy.sections.security.title', 'Data Security'), 'content' => ai_get_text('legal.privacy.sections.security.content', 'We implement technical and organizational measures to protect your personal information from unauthorized access.')],
            ['title' => ai_get_text('legal.privacy.sections.rights.title', 'Your Rights'), 'content' => ai_get_text('legal.privacy.sections.rights.content', 'You have the right to access, correct, or delete your personal data. You may also withdraw your consent to data processing at any time.')],
            ['title' => ai_get_text('legal.privacy.sections.retention.title', 'Data Retention'), 'content' => ai_get_text('legal.privacy.sections.retention.content', 'We retain your personal information only for as long as necessary for the purposes described in this policy.')],
            ['title' => ai_get_text('legal.privacy.sections.changes.title', 'Changes to This Policy'), 'content' => ai_get_text('legal.privacy.sections.changes.content', 'We may update this Privacy Policy periodically. We will notify you of any significant changes by posting a new version on this page.')],
        ];

        $title = !empty($s['title']) ? $s['title'] : $t['title'];
        $companyName = !empty($s['company_name']) ? $s['company_name'] : (!empty($s['legal_name']) ? $s['legal_name'] : ai_get_text('common.ourCompany', 'Our company'));

        // Generate email from company name if not provided (same logic as footer)
        $email = !empty($s['contact_email']) ? $s['contact_email'] : '';
        if (empty($email) || $email === 'privacy@example.com') {
            $brandSlug = sanitize_title($companyName);
            $brandSlug = preg_replace('/[^a-z0-9]/', '', $brandSlug);
            $email = $brandSlug ? "info@{$brandSlug}.com" : 'info@company.com';
        }

        $lastUpdated = !empty($s['last_updated']) ? $s['last_updated'] : date_i18n('Y-m-d');

        // Use custom sections if provided, otherwise default
        $sections = $defaultSections;
        if (!empty($s['sections'])) {
            $sections = [];
            foreach ($s['sections'] as $section) {
                $sections[] = ['title' => $section['section_title'], 'content' => $section['section_content']];
            }
        }

        // Check if company details exist
        $hasCompanyDetails = !empty($s['legal_name']) || !empty($s['address']) || !empty($s['inn']) || !empty($s['ogrn']);

        $introTemplate = ai_get_text('legal.privacy.introduction', '{company} respects your privacy and is committed to protecting your personal data. This Privacy Policy explains what information we collect, how we use it, and what measures we take to protect it.');
        $introduction = str_replace('{company}', $companyName, $introTemplate);
        ?>
        <section class="ai-privacy ai-privacy-default py-16 px-4 section-light" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="privacy" data-block-variant="privacy-default">
            <div class="ai-container-narrow">
                <div class="mb-12 text-center">
                    <h1 class="text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($title); ?></h1>
                    <p class="text-[var(--text-muted)]"><?php echo esc_html($t['lastUpdated']); ?>: <?php echo esc_html($lastUpdated); ?></p>
                </div>

                <div class="mb-10 p-6 bg-[var(--bg-secondary)] rounded-xl">
                    <p class="text-[var(--text-primary)] leading-relaxed"><?php echo esc_html($introduction); ?></p>
                </div>

                <div class="space-y-8">
                    <?php foreach ($sections as $index => $section) : ?>
                    <div class="border-b border-[var(--border-color)] pb-8 last:border-0">
                        <h2 class="text-2xl font-semibold text-[var(--text-primary)] mb-4"><?php echo esc_html(($index + 1) . '. ' . $section['title']); ?></h2>
                        <p class="text-[var(--text-secondary)] leading-relaxed"><?php echo esc_html($section['content']); ?></p>
                    </div>
                    <?php endforeach; ?>
                </div>

                <?php if ($hasCompanyDetails) : ?>
                <div class="mt-12 p-6 bg-[var(--bg-secondary)] rounded-xl border border-[var(--border-color)]">
                    <h2 class="text-xl font-semibold text-[var(--text-primary)] mb-3"><?php echo esc_html($t['operatorTitle']); ?></h2>
                    <p class="text-[var(--text-secondary)] mb-4"><?php echo esc_html($t['operatorText']); ?></p>
                    <div class="space-y-2 text-[var(--text-primary)]">
                        <?php if (!empty($s['legal_name'])) : ?><p class="font-medium"><?php echo esc_html($s['legal_name']); ?></p><?php endif; ?>
                        <?php if (!empty($s['address'])) : ?><p><span class="text-[var(--text-muted)]"><?php echo esc_html($t['addressLabel']); ?>:</span> <?php echo esc_html($s['address']); ?></p><?php endif; ?>
                        <?php if (!empty($s['inn'])) : ?><p><span class="text-[var(--text-muted)]"><?php echo esc_html($t['innLabel']); ?>:</span> <?php echo esc_html($s['inn']); ?></p><?php endif; ?>
                        <?php if (!empty($s['ogrn'])) : ?><p><span class="text-[var(--text-muted)]"><?php echo esc_html($t['ogrnLabel']); ?>:</span> <?php echo esc_html($s['ogrn']); ?></p><?php endif; ?>
                        <?php if (!empty($s['director'])) : ?><p><span class="text-[var(--text-muted)]"><?php echo esc_html($t['directorLabel']); ?>:</span> <?php echo esc_html($s['director']); ?></p><?php endif; ?>
                        <?php if (!empty($s['phone'])) : ?><p><span class="text-[var(--text-muted)]"><?php echo esc_html($t['phoneLabel']); ?>:</span> <?php echo esc_html($s['phone']); ?></p><?php endif; ?>
                        <?php if (!empty($s['contact_email'])) : ?><p><span class="text-[var(--text-muted)]"><?php echo esc_html($t['emailLabel']); ?>:</span> <a href="mailto:<?php echo esc_attr($s['contact_email']); ?>" class="text-[var(--primary)] hover:opacity-80"><?php echo esc_html($s['contact_email']); ?></a></p><?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>

                <div class="mt-12 p-6 bg-[var(--primary-light,#dbeafe)] rounded-xl">
                    <h2 class="text-xl font-semibold text-[var(--text-primary)] mb-3"><?php echo esc_html($t['contactTitle']); ?></h2>
                    <p class="text-[var(--text-secondary)]"><?php echo esc_html($t['contactText']); ?></p>
                    <p class="mt-3"><a href="mailto:<?php echo esc_attr($email); ?>" class="text-[var(--primary)] hover:opacity-80 font-medium"><?php echo esc_html($email); ?></a></p>
                </div>
            </div>
        </section>
        <?php
    }
}
