<?php
/**
 * Product Detail Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Product Detail - Full product page layout with tabs
 */
class Product_Detail extends AI_Widget_Base {

    public function get_name() {
        return 'ai_product_detail';
    }

    public function get_title() {
        return __('Product Detail', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-single-product';
    }

    public function get_categories() {
        return array('ai-theme-product');
    }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('product_id', array(
            'label' => __('Product ID', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'description' => __('Leave empty for current product', 'ai-theme'),
        ));

        $this->add_control('add_to_cart_text', array(
            'label' => __('Add to Cart Text', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('buy_now_text', array(
            'label' => __('Buy Now Text', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('show_specifications', array(
            'label' => __('Show Specifications', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ));

        $this->add_control('show_reviews', array(
            'label' => __('Show Reviews', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ));

        $this->add_control('show_stock', array(
            'label' => __('Show Stock Indicator', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        $product_id = !empty($settings['product_id']) ? $settings['product_id'] : get_the_ID();
        $product = wc_get_product($product_id);

        if (!$product) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('Product not found', 'ai-theme') . '</p>';
            return;
        }

        $image_id = $product->get_image_id();
        $gallery_ids = $product->get_gallery_image_ids();
        $all_images = array_merge(array($image_id), $gallery_ids);
        $is_on_sale = $product->is_on_sale();
        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();
        $discount_percent = 0;
        if ($is_on_sale && $regular_price && $sale_price) {
            $discount_percent = round((($regular_price - $sale_price) / $regular_price) * 100);
        }
        $widget_id = $this->get_id();
        ?>
        <div class="w-full bg-[var(--bg-primary)] py-12" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="product" data-block-variant="product-detail">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-12">
                    <!-- Left Column - Gallery -->
                    <div class="space-y-4">
                        <!-- Main Image -->
                        <div class="relative aspect-square bg-[var(--bg-secondary)] rounded-xl overflow-hidden">
                            <?php if ($image_id) : ?>
                                <img src="<?php echo esc_url(wp_get_attachment_image_url($image_id, 'large')); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover gallery-main-<?php echo esc_attr($widget_id); ?>" />
                            <?php endif; ?>
                            <?php if ($discount_percent > 0) : ?>
                                <div class="absolute top-4 right-4 bg-red-500 text-white px-3 py-1 rounded-full font-bold text-sm">
                                    -<?php echo esc_html($discount_percent); ?>%
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Thumbnails -->
                        <?php if (count($all_images) > 1) : ?>
                            <div class="grid grid-cols-4 gap-3">
                                <?php foreach ($all_images as $idx => $img_id) :
                                    if (!$img_id) continue;
                                    $thumb_url = wp_get_attachment_image_url($img_id, 'thumbnail');
                                    $large_url = wp_get_attachment_image_url($img_id, 'large');
                                ?>
                                    <button
                                        class="aspect-square bg-[var(--bg-secondary)] rounded-lg overflow-hidden border-2 transition-all gallery-thumb-<?php echo esc_attr($widget_id); ?> <?php echo $idx === 0 ? 'border-[var(--primary)] ring-2 ring-[var(--primary-light,#dbeafe)]' : 'border-transparent hover:border-[var(--border-color)]'; ?>"
                                        data-image="<?php echo esc_url($large_url); ?>"
                                    >
                                        <img src="<?php echo esc_url($thumb_url); ?>" alt="" class="w-full h-full object-cover" />
                                    </button>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Right Column - Product Info -->
                    <div class="space-y-6">
                        <!-- Title and Rating -->
                        <div>
                            <?php 
                            $brand = '';
                            if (taxonomy_exists('pa_brand')) {
                                $brands = wp_get_post_terms($product_id, 'pa_brand');
                                if (!empty($brands) && !is_wp_error($brands)) {
                                    $brand = $brands[0]->name;
                                }
                            }
                            if ($brand) : ?>
                                <p class="text-sm text-[var(--text-muted)] mb-2"><?php echo esc_html($brand); ?></p>
                            <?php endif; ?>
                            <h1 class="text-3xl font-bold text-[var(--text-primary)] mb-2"><?php echo esc_html($product->get_name()); ?></h1>
                            <?php if ($product->get_average_rating()) : ?>
                                <div class="flex items-center gap-2">
                                    <div class="flex items-center gap-1">
                                        <?php for ($i = 1; $i <= 5; $i++) : ?>
                                            <svg class="w-5 h-5 <?php echo $i <= round($product->get_average_rating()) ? 'text-yellow-400 fill-yellow-400' : 'text-[var(--border-color)]'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                        <?php endfor; ?>
                                    </div>
                                    <span class="text-sm text-[var(--text-secondary)]">
                                        <?php echo esc_html($product->get_average_rating()); ?> (<?php echo esc_html($product->get_review_count()); ?> <?php _e('reviews', 'ai-theme'); ?>)
                                    </span>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Price -->
                        <div class="flex items-baseline gap-3">
                            <?php if ($is_on_sale) : ?>
                                <span class="text-4xl font-bold text-[var(--text-primary)]"><?php echo wc_price($sale_price); ?></span>
                                <span class="text-xl text-[var(--text-muted)] line-through"><?php echo wc_price($regular_price); ?></span>
                            <?php else : ?>
                                <span class="text-4xl font-bold text-[var(--text-primary)]"><?php echo $product->get_price_html(); ?></span>
                            <?php endif; ?>
                        </div>

                        <!-- Short Description -->
                        <?php if ($product->get_short_description()) : ?>
                            <p class="text-[var(--text-secondary)] text-lg"><?php echo wp_strip_all_tags($product->get_short_description()); ?></p>
                        <?php endif; ?>

                        <!-- Stock Indicator -->
                        <?php if ($settings['show_stock'] === 'yes') : ?>
                            <div class="flex items-center gap-2">
                                <?php if ($product->is_in_stock()) : ?>
                                    <svg class="w-5 h-5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>
                                    <span class="text-green-600 font-medium"><?php _e('In stock', 'ai-theme'); ?></span>
                                    <?php if ($product->get_stock_quantity() && $product->get_stock_quantity() <= 10) : ?>
                                        <span class="text-sm text-[var(--text-muted)]">
                                            (<?php echo esc_html($product->get_stock_quantity()); ?> <?php _e('left', 'ai-theme'); ?>)
                                        </span>
                                    <?php endif; ?>
                                <?php else : ?>
                                    <span class="text-red-600 font-medium"><?php _e('Out of stock', 'ai-theme'); ?></span>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>

                        <!-- Variations -->
                        <?php if ($product->is_type('variable')) :
                            $attributes = $product->get_variation_attributes();
                            foreach ($attributes as $attribute_name => $options) :
                                $label = wc_attribute_label($attribute_name);
                        ?>
                            <div>
                                <label class="block text-sm font-semibold text-[var(--text-primary)] mb-3">
                                    <?php echo esc_html($label); ?> <span class="text-red-500">*</span>
                                </label>
                                <div class="flex gap-2 flex-wrap">
                                    <?php foreach ($options as $option) : ?>
                                        <button class="px-6 py-3 rounded-lg border-2 border-[var(--border-color)] font-medium transition-all hover:border-[var(--text-muted)] variation-btn" data-attribute="<?php echo esc_attr($attribute_name); ?>" data-value="<?php echo esc_attr($option); ?>">
                                            <?php echo esc_html($option); ?>
                                        </button>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endforeach; endif; ?>

                        <!-- Quantity -->
                        <div>
                            <label class="block text-sm font-semibold text-[var(--text-primary)] mb-3">
                                <?php _e('Quantity', 'ai-theme'); ?>
                            </label>
                            <div class="flex items-center gap-4">
                                <div class="flex items-center border-2 border-[var(--border-color)] rounded-lg">
                                    <button class="p-3 hover:bg-[var(--bg-secondary)] transition-colors qty-minus" type="button">
                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4"/></svg>
                                    </button>
                                    <input type="number" class="qty-input w-12 text-center font-semibold text-lg border-0 bg-transparent" value="1" min="1" max="<?php echo esc_attr($product->get_stock_quantity() ?: 999); ?>" />
                                    <button class="p-3 hover:bg-[var(--bg-secondary)] transition-colors qty-plus" type="button">
                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Action Buttons -->
                        <div class="space-y-3">
                            <a href="<?php echo esc_url($product->add_to_cart_url()); ?>" class="w-full bg-[var(--primary)] text-white py-4 px-6 rounded-lg font-semibold hover:opacity-90 transition-colors flex items-center justify-center gap-2 <?php echo !$product->is_in_stock() ? 'opacity-50 pointer-events-none' : ''; ?>">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                                <?php echo esc_html($settings['add_to_cart_text']); ?>
                            </a>

                            <div class="grid grid-cols-2 gap-3">
                                <a href="<?php echo esc_url(wc_get_checkout_url() . '?add-to-cart=' . $product_id); ?>" class="border-2 border-[var(--primary)] text-[var(--primary)] py-3 px-6 rounded-lg font-semibold hover:bg-[var(--primary-light,#dbeafe)] transition-colors text-center <?php echo !$product->is_in_stock() ? 'opacity-50 pointer-events-none' : ''; ?>">
                                    <?php echo esc_html($settings['buy_now_text']); ?>
                                </a>
                                <button class="border-2 border-[var(--border-color)] py-3 px-6 rounded-lg font-semibold transition-colors flex items-center justify-center gap-2 hover:bg-[var(--bg-secondary)] wishlist-btn">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/></svg>
                                    <?php _e('Wishlist', 'ai-theme'); ?>
                                </button>
                            </div>
                        </div>

                        <!-- Benefits -->
                        <div class="border-t border-[var(--border-color)] pt-6 space-y-3">
                            <div class="flex items-center gap-3 text-sm text-[var(--text-secondary)]">
                                <svg class="w-5 h-5 text-[var(--text-muted)]" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 17H5a2 2 0 01-2-2V5a2 2 0 012-2h14a2 2 0 012 2v10a2 2 0 01-2 2h-3m-1 4l-4 4m0 0l-4-4m4 4V9"/></svg>
                                <span><?php _e('Free shipping', 'ai-theme'); ?></span>
                            </div>
                            <div class="flex items-center gap-3 text-sm text-[var(--text-secondary)]">
                                <svg class="w-5 h-5 text-[var(--text-muted)]" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"/></svg>
                                <span><?php _e('Money back guarantee', 'ai-theme'); ?></span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Tabs -->
                <div class="mt-16">
                    <div class="border-b border-[var(--border-color)]">
                        <div class="flex gap-8">
                            <button class="tab-btn pb-4 px-2 font-semibold border-b-2 border-[var(--primary)] text-[var(--primary)]" data-tab="description">
                                <?php _e('Description', 'ai-theme'); ?>
                            </button>
                            <?php if ($settings['show_specifications'] === 'yes') : ?>
                                <button class="tab-btn pb-4 px-2 font-semibold border-b-2 border-transparent text-[var(--text-muted)] hover:text-[var(--text-primary)]" data-tab="specifications">
                                    <?php _e('Specifications', 'ai-theme'); ?>
                                </button>
                            <?php endif; ?>
                            <?php if ($settings['show_reviews'] === 'yes') : ?>
                                <button class="tab-btn pb-4 px-2 font-semibold border-b-2 border-transparent text-[var(--text-muted)] hover:text-[var(--text-primary)]" data-tab="reviews">
                                    <?php _e('Reviews', 'ai-theme'); ?> (<?php echo esc_html($product->get_review_count()); ?>)
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="py-8">
                        <!-- Description Tab -->
                        <div class="tab-content" id="tab-description">
                            <div class="prose max-w-none text-[var(--text-primary)] text-lg leading-relaxed">
                                <?php echo wp_kses_post($product->get_description()); ?>
                            </div>
                        </div>

                        <!-- Specifications Tab -->
                        <?php if ($settings['show_specifications'] === 'yes') : ?>
                            <div class="tab-content hidden" id="tab-specifications">
                                <div class="max-w-3xl">
                                    <?php 
                                    $attributes = $product->get_attributes();
                                    if (!empty($attributes)) : ?>
                                        <div class="divide-y divide-[var(--border-color)]">
                                            <?php foreach ($attributes as $attribute) : 
                                                if (!$attribute->get_visible()) continue;
                                                $name = wc_attribute_label($attribute->get_name());
                                                $values = $attribute->is_taxonomy() 
                                                    ? implode(', ', wc_get_product_terms($product_id, $attribute->get_name(), array('fields' => 'names')))
                                                    : implode(', ', $attribute->get_options());
                                            ?>
                                                <div class="py-4 flex justify-between">
                                                    <span class="font-medium text-[var(--text-primary)]"><?php echo esc_html($name); ?></span>
                                                    <span class="text-[var(--text-secondary)]"><?php echo esc_html($values); ?></span>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php else : ?>
                                        <p class="text-[var(--text-muted)]"><?php _e('No specifications', 'ai-theme'); ?></p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>

                        <!-- Reviews Tab -->
                        <?php if ($settings['show_reviews'] === 'yes') : ?>
                            <div class="tab-content hidden" id="tab-reviews">
                                <?php comments_template(); ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <script>
        (function() {
            const primary = getComputedStyle(document.documentElement).getPropertyValue('--primary').trim() || '#2563eb';
            const primaryLight = getComputedStyle(document.documentElement).getPropertyValue('--primary-light').trim() || '#dbeafe';

            // Gallery
            const mainImg = document.querySelector('.gallery-main-<?php echo esc_attr($widget_id); ?>');
            const thumbs = document.querySelectorAll('.gallery-thumb-<?php echo esc_attr($widget_id); ?>');
            thumbs.forEach(thumb => {
                thumb.addEventListener('click', function() {
                    if (mainImg) mainImg.src = this.dataset.image;
                    thumbs.forEach(t => {
                        t.style.borderColor = 'transparent';
                        t.style.boxShadow = 'none';
                    });
                    this.style.borderColor = primary;
                    this.style.boxShadow = '0 0 0 2px ' + primaryLight;
                });
            });

            // Quantity
            const qtyInput = document.querySelector('.qty-input');
            const qtyMinus = document.querySelector('.qty-minus');
            const qtyPlus = document.querySelector('.qty-plus');
            if (qtyMinus && qtyInput) {
                qtyMinus.addEventListener('click', () => {
                    const val = parseInt(qtyInput.value) || 1;
                    if (val > 1) qtyInput.value = val - 1;
                });
            }
            if (qtyPlus && qtyInput) {
                qtyPlus.addEventListener('click', () => {
                    const val = parseInt(qtyInput.value) || 1;
                    const max = parseInt(qtyInput.max) || 999;
                    if (val < max) qtyInput.value = val + 1;
                });
            }

            // Tabs
            const tabBtns = document.querySelectorAll('.tab-btn');
            const tabContents = document.querySelectorAll('.tab-content');
            tabBtns.forEach(btn => {
                btn.addEventListener('click', function() {
                    const tab = this.dataset.tab;
                    tabBtns.forEach(b => {
                        b.style.borderColor = 'transparent';
                        b.style.color = 'var(--text-muted)';
                    });
                    this.style.borderColor = primary;
                    this.style.color = primary;
                    tabContents.forEach(c => c.classList.add('hidden'));
                    document.getElementById('tab-' + tab)?.classList.remove('hidden');
                });
            });
        })();
        </script>
        <?php
    }
}
