<?php
/**
 * Product Gallery Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Product Gallery Widget - Full-width gallery focused layout
 */
class Product_Gallery extends AI_Widget_Base {

    public function get_name() {
        return 'ai_product_gallery';
    }

    public function get_title() {
        return __('Product Gallery', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-single-product';
    }

    public function get_categories() {
        return array('ai-theme-shop');
    }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('product_id', array(
            'label' => __('Product ID', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
        ));

        $this->add_control('add_to_cart_text', array(
            'label' => __('Add to Cart Text', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) return;

        $product_id = !empty($settings['product_id']) ? $settings['product_id'] : get_the_ID();
        $product = wc_get_product($product_id);
        if (!$product) return;

        $image_id = $product->get_image_id();
        $gallery_ids = $product->get_gallery_image_ids();
        $all_images = array_merge(array($image_id), $gallery_ids);
        $all_images = array_filter($all_images);
        
        $is_on_sale = $product->is_on_sale();
        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();
        $discount_percent = 0;
        if ($is_on_sale && $regular_price && $sale_price) {
            $discount_percent = round((($regular_price - $sale_price) / $regular_price) * 100);
        }
        
        $widget_id = $this->get_id();
        ?>
        <section class="bg-[var(--bg-primary)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="product" data-block-variant="product-gallery">
            <!-- Full-width Gallery -->
            <div class="relative aspect-[16/9] md:aspect-[21/9] bg-[var(--bg-secondary)] gallery-container-<?php echo esc_attr($widget_id); ?>">
                <?php foreach ($all_images as $idx => $img_id) : 
                    $url = wp_get_attachment_image_url($img_id, 'full');
                ?>
                    <img 
                        src="<?php echo esc_url($url); ?>" 
                        alt="<?php echo esc_attr($product->get_name()); ?>"
                        class="absolute inset-0 w-full h-full object-contain gallery-slide-<?php echo esc_attr($widget_id); ?> <?php echo $idx === 0 ? '' : 'hidden'; ?>"
                        data-index="<?php echo esc_attr($idx); ?>"
                    />
                <?php endforeach; ?>

                <!-- Navigation Arrows -->
                <?php if (count($all_images) > 1) : ?>
                    <button class="absolute left-4 top-1/2 -translate-y-1/2 w-12 h-12 bg-[var(--bg-primary)]/80 rounded-full flex items-center justify-center hover:bg-[var(--bg-primary)] shadow-lg gallery-prev-<?php echo esc_attr($widget_id); ?>">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/></svg>
                    </button>
                    <button class="absolute right-4 top-1/2 -translate-y-1/2 w-12 h-12 bg-[var(--bg-primary)]/80 rounded-full flex items-center justify-center hover:bg-[var(--bg-primary)] shadow-lg gallery-next-<?php echo esc_attr($widget_id); ?>">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/></svg>
                    </button>
                <?php endif; ?>

                <!-- Discount Badge -->
                <?php if ($discount_percent > 0) : ?>
                    <div class="absolute top-4 left-4 bg-red-500 text-white px-3 py-1 rounded-full font-bold">
                        -<?php echo esc_html($discount_percent); ?>%
                    </div>
                <?php endif; ?>

                <!-- Dots -->
                <?php if (count($all_images) > 1) : ?>
                    <div class="absolute bottom-4 left-1/2 -translate-x-1/2 flex gap-2">
                        <?php foreach ($all_images as $idx => $img_id) : ?>
                            <button 
                                class="gallery-dot-<?php echo esc_attr($widget_id); ?> h-2 rounded-full transition-all <?php echo $idx === 0 ? 'w-8 bg-[var(--primary)]' : 'w-2 bg-[var(--bg-primary)]/50'; ?>"
                                data-index="<?php echo esc_attr($idx); ?>"
                            ></button>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Product Info -->
            <div class="max-w-4xl mx-auto px-4 py-8">
                <div class="flex flex-col md:flex-row md:items-start md:justify-between gap-6">
                    <div class="flex-1">
                        <?php 
                        $brand = '';
                        if (taxonomy_exists('pa_brand')) {
                            $brands = wp_get_post_terms($product_id, 'pa_brand');
                            if (!empty($brands) && !is_wp_error($brands)) {
                                $brand = $brands[0]->name;
                            }
                        }
                        if ($brand) : ?>
                            <p class="text-sm text-[var(--text-muted)] mb-1"><?php echo esc_html($brand); ?></p>
                        <?php endif; ?>
                        <h1 class="text-3xl font-bold text-[var(--text-primary)] mb-2"><?php echo esc_html($product->get_name()); ?></h1>

                        <?php if ($product->get_average_rating()) : ?>
                            <div class="flex items-center gap-2 mb-4">
                                <div class="flex">
                                    <?php for ($i = 1; $i <= 5; $i++) : ?>
                                        <svg class="w-5 h-5 <?php echo $i <= round($product->get_average_rating()) ? 'text-yellow-400 fill-yellow-400' : 'text-[var(--border-color)]'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                    <?php endfor; ?>
                                </div>
                                <span class="text-[var(--text-muted)]">(<?php echo esc_html($product->get_review_count()); ?> reviews)</span>
                            </div>
                        <?php endif; ?>

                        <?php if ($product->get_short_description()) : ?>
                            <p class="text-[var(--text-secondary)] mb-4"><?php echo wp_strip_all_tags($product->get_short_description()); ?></p>
                        <?php endif; ?>
                    </div>

                    <div class="md:text-right">
                        <div class="flex items-baseline gap-3 mb-4">
                            <?php if ($is_on_sale) : ?>
                                <span class="text-4xl font-bold text-[var(--text-primary)]"><?php echo wc_price($sale_price); ?></span>
                                <span class="text-xl text-[var(--text-muted)] line-through"><?php echo wc_price($regular_price); ?></span>
                            <?php else : ?>
                                <span class="text-4xl font-bold text-[var(--text-primary)]"><?php echo $product->get_price_html(); ?></span>
                            <?php endif; ?>
                        </div>

                        <div class="flex gap-3">
                            <a href="<?php echo esc_url($product->add_to_cart_url()); ?>" class="flex-1 bg-[var(--primary)] text-white py-3 px-6 rounded-[var(--btn-radius)] font-semibold hover:opacity-90 flex items-center justify-center gap-2 <?php echo !$product->is_in_stock() ? 'opacity-50 pointer-events-none' : ''; ?>">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                                <?php echo esc_html($settings['add_to_cart_text']); ?>
                            </a>
                            <button class="p-3 border border-[var(--border-color)] rounded-[var(--btn-radius)] hover:bg-[var(--bg-secondary)]">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/></svg>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </section>

        <script>
        (function() {
            const slides = document.querySelectorAll('.gallery-slide-<?php echo esc_attr($widget_id); ?>');
            const dots = document.querySelectorAll('.gallery-dot-<?php echo esc_attr($widget_id); ?>');
            const prevBtn = document.querySelector('.gallery-prev-<?php echo esc_attr($widget_id); ?>');
            const nextBtn = document.querySelector('.gallery-next-<?php echo esc_attr($widget_id); ?>');
            
            if (slides.length <= 1) return;
            
            let current = 0;
            const total = slides.length;

            function showSlide(index) {
                slides.forEach((s, i) => {
                    s.classList.toggle('hidden', i !== index);
                });
                dots.forEach((d, i) => {
                    d.classList.toggle('w-8', i === index);
                    d.classList.toggle('bg-[var(--primary)]', i === index);
                    d.classList.toggle('w-2', i !== index);
                    d.classList.toggle('bg-[var(--bg-primary)]/50', i !== index);
                });
                current = index;
            }

            if (prevBtn) {
                prevBtn.addEventListener('click', () => showSlide((current - 1 + total) % total));
            }
            if (nextBtn) {
                nextBtn.addEventListener('click', () => showSlide((current + 1) % total));
            }
            dots.forEach(dot => {
                dot.addEventListener('click', function() {
                    showSlide(parseInt(this.dataset.index));
                });
            });
        })();
        </script>
        <?php
    }
}
