<?php
/**
 * Products Grid Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit;

/**
 * Products Grid - Standard product grid layout
 */
class Products_Grid extends AI_Widget_Base {
    public function get_name() { return 'ai_products_grid'; }
    public function get_title() { return __('Products Grid', 'ai-theme'); }
    public function get_icon() { return 'eicon-products'; }
    public function get_categories() { return array('ai-theme-products'); }

    protected function register_controls() {
        $this->start_controls_section('content_section', array(
            'label' => __('Content', 'ai-theme'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ));

        $this->add_control('heading', array(
            'label' => __('Heading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
        ));

        $this->add_control('subheading', array(
            'label' => __('Subheading', 'ai-theme'),
            'type' => Controls_Manager::TEXT,
        ));

        $this->add_control('posts_per_page', array(
            'label' => __('Number of Products', 'ai-theme'),
            'type' => Controls_Manager::NUMBER,
            'default' => 8,
            'min' => 1,
            'max' => 24,
        ));

        $this->add_control('orderby', array(
            'label' => __('Order By', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => 'date',
            'options' => array(
                'date' => __('Date', 'ai-theme'),
                'title' => __('Title', 'ai-theme'),
                'price' => __('Price', 'ai-theme'),
                'popularity' => __('Popularity', 'ai-theme'),
                'rating' => __('Rating', 'ai-theme'),
                'rand' => __('Random', 'ai-theme'),
            ),
        ));

        $categories = $this->get_product_categories();
        $this->add_control('category', array(
            'label' => __('Category', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'default' => '',
            'options' => $categories,
        ));

        $this->add_control('columns', array(
            'label' => __('Columns', 'ai-theme'),
            'type' => Controls_Manager::SELECT,
            'options' => array('2' => '2', '3' => '3', '4' => '4'),
            'default' => '4',
        ));

        $this->add_control('show_rating', array(
            'label' => __('Show Rating', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ));

        $this->add_control('show_category', array(
            'label' => __('Show Category', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ));

        $this->add_control('show_add_to_cart', array(
            'label' => __('Show Add to Cart', 'ai-theme'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ));

        $this->end_controls_section();
    }

    private function get_product_categories() {
        $categories = array('' => __('All Categories', 'ai-theme'));
        if (!function_exists('get_terms')) return $categories;

        $terms = get_terms(array(
            'taxonomy' => 'product_cat',
            'hide_empty' => true,
        ));

        if (!is_wp_error($terms) && !empty($terms)) {
            foreach ($terms as $term) {
                $categories[$term->slug] = $term->name;
            }
        }

        return $categories;
    }

    protected function render() {
        $s = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');

        if (!class_exists('WooCommerce')) {
            echo '<p class="text-center text-[var(--text-muted)] py-8">' . __('WooCommerce is required', 'ai-theme') . '</p>';
            return;
        }

        $grid_cols = array(
            '2' => 'grid-cols-1 md:grid-cols-2',
            '3' => 'grid-cols-1 md:grid-cols-2 lg:grid-cols-3',
            '4' => 'grid-cols-1 md:grid-cols-2 lg:grid-cols-4'
        );
        $grid_class = $grid_cols[$s['columns']] ?? 'grid-cols-1 md:grid-cols-2 lg:grid-cols-4';

        $args = array(
            'status' => 'publish',
            'limit' => $s['posts_per_page'] ?? 8,
            'orderby' => $s['orderby'] ?? 'date',
            'order' => 'DESC',
        );
        if (!empty($s['category'])) {
            $args['category'] = array($s['category']);
        }
        $products = wc_get_products($args);
        ?>
        <div class="w-full bg-[var(--bg-secondary)] py-16" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="products" data-block-variant="products-grid">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <?php if (!empty($s['heading']) || !empty($s['subheading'])) : ?>
                    <div class="text-center mb-12">
                        <?php if (!empty($s['heading'])) : ?>
                            <h2 class="text-3xl md:text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($s['heading']); ?></h2>
                        <?php endif; ?>
                        <?php if (!empty($s['subheading'])) : ?>
                            <p class="text-lg text-[var(--text-secondary)] max-w-2xl mx-auto"><?php echo esc_html($s['subheading']); ?></p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <?php if (empty($products)) : ?>
                    <div class="text-center py-16">
                        <svg class="w-16 h-16 text-[var(--text-muted)] mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                        <h3 class="text-xl font-semibold text-[var(--text-primary)] mb-2"><?php _e('No products', 'ai-theme'); ?></h3>
                        <p class="text-[var(--text-secondary)]"><?php _e('Add products in WooCommerce', 'ai-theme'); ?></p>
                    </div>
                <?php else : ?>
                    <div class="grid <?php echo esc_attr($grid_class); ?> gap-6 mb-12">
                        <?php foreach ($products as $product) : 
                            $image_id = $product->get_image_id();
                            $image_url = $image_id ? wp_get_attachment_image_url($image_id, 'woocommerce_thumbnail') : wc_placeholder_img_src();
                            $rating = $product->get_average_rating();
                            $is_on_sale = $product->is_on_sale();
                            $discount = 0;
                            if ($is_on_sale && $product->get_regular_price() && $product->get_sale_price()) {
                                $discount = round((1 - $product->get_sale_price() / $product->get_regular_price()) * 100);
                            }
                            $categories = wc_get_product_category_list($product->get_id(), ', ');
                        ?>
                            <div class="group bg-[var(--bg-primary)] rounded-xl overflow-hidden shadow-sm hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1">
                                <!-- Image -->
                                <div class="relative aspect-square overflow-hidden bg-[var(--bg-secondary)]">
                                    <a href="<?php echo esc_url($product->get_permalink()); ?>">
                                        <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($product->get_name()); ?>" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500" />
                                    </a>

                                    <?php if ($discount > 0) : ?>
                                        <div class="absolute top-4 left-4 bg-red-500 text-white px-3 py-1 rounded-full font-bold text-sm">
                                            -<?php echo esc_html($discount); ?>%
                                        </div>
                                    <?php endif; ?>

                                    <?php if (!$product->is_in_stock()) : ?>
                                        <div class="absolute inset-0 bg-black bg-opacity-50 flex items-center justify-center">
                                            <span class="bg-[var(--bg-primary)] text-[var(--text-primary)] px-4 py-2 rounded-lg font-semibold">
                                                <?php _e('Out of stock', 'ai-theme'); ?>
                                            </span>
                                        </div>
                                    <?php endif; ?>

                                    <!-- Wishlist Button -->
                                    <div class="absolute top-4 right-4 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                                        <button class="p-2 bg-[var(--bg-primary)] rounded-full shadow-md hover:bg-red-50 transition-colors">
                                            <svg class="w-5 h-5 text-[var(--text-primary)] hover:text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/></svg>
                                        </button>
                                    </div>

                                    <?php if ($s['show_add_to_cart'] === 'yes' && $product->is_in_stock()) : ?>
                                        <div class="absolute bottom-4 left-4 right-4 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                                            <a href="<?php echo esc_url($product->add_to_cart_url()); ?>" class="w-full bg-[var(--primary)] text-white py-2 px-4 rounded-lg font-semibold hover:opacity-90 transition-colors flex items-center justify-center gap-2">
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/></svg>
                                                <?php _e('Add to cart', 'ai-theme'); ?>
                                            </a>
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <!-- Info -->
                                <div class="p-4">
                                    <?php if ($s['show_category'] === 'yes' && $categories) : ?>
                                        <p class="text-xs text-[var(--text-muted)] uppercase tracking-wide mb-2">
                                            <?php echo wp_strip_all_tags($categories); ?>
                                        </p>
                                    <?php endif; ?>

                                    <a href="<?php echo esc_url($product->get_permalink()); ?>" class="block">
                                        <h3 class="text-lg font-semibold text-[var(--text-primary)] mb-2 line-clamp-2 group-hover:text-[var(--primary)] transition-colors">
                                            <?php echo esc_html($product->get_name()); ?>
                                        </h3>
                                    </a>

                                    <?php if ($s['show_rating'] === 'yes' && $rating > 0) : ?>
                                        <div class="flex items-center gap-2 mb-3">
                                            <div class="flex items-center gap-1">
                                                <?php for ($i = 1; $i <= 5; $i++) : ?>
                                                    <svg class="w-4 h-4 <?php echo $i <= round($rating) ? 'text-yellow-400 fill-yellow-400' : 'text-[var(--border-color)]'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                                <?php endfor; ?>
                                            </div>
                                            <span class="text-sm text-[var(--text-secondary)]">(<?php echo esc_html($product->get_review_count()); ?>)</span>
                                        </div>
                                    <?php endif; ?>

                                    <div class="flex items-baseline gap-2">
                                        <span class="text-xl font-bold text-[var(--text-primary)]"><?php echo $product->get_price_html(); ?></span>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
}
