<?php
/**
 * Reviews Slider Widget
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Reviews Slider Widget - Carousel/slider layout
 */
class Reviews_Slider extends AI_Widget_Base {

    public function get_name() {
        return 'ai_reviews_slider';
    }

    public function get_title() {
        return __('Reviews Slider', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-review';
    }

    public function get_categories() {
        return array('ai-theme-testimonials');
    }

    protected function register_controls() {
        $this->start_controls_section('content_section', array('label' => __('Content', 'ai-theme'), 'tab' => Controls_Manager::TAB_CONTENT));

        $this->add_control('heading', array('label' => __('Heading', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => ''));
        $this->add_control('subheading', array('label' => __('Subheading', 'ai-theme'), 'type' => Controls_Manager::TEXTAREA));
        $this->add_control('autoplay', array('label' => __('Autoplay', 'ai-theme'), 'type' => Controls_Manager::SWITCHER, 'default' => 'yes'));
        $this->add_control('autoplay_speed', array('label' => __('Autoplay Speed (ms)', 'ai-theme'), 'type' => Controls_Manager::NUMBER, 'default' => 5000));

        $repeater = new Repeater();
        $repeater->add_control('author', array('label' => __('Author', 'ai-theme'), 'type' => Controls_Manager::TEXT, 'default' => 'Customer'));
        $repeater->add_control('role', array('label' => __('Role', 'ai-theme'), 'type' => Controls_Manager::TEXT));
        $repeater->add_control('comment', array('label' => __('Comment', 'ai-theme'), 'type' => Controls_Manager::TEXTAREA, 'default' => 'Great product!'));
        $repeater->add_control('rating', array('label' => __('Rating', 'ai-theme'), 'type' => Controls_Manager::NUMBER, 'min' => 1, 'max' => 5, 'default' => 5));
        $repeater->add_control('avatar', array('label' => __('Avatar', 'ai-theme'), 'type' => Controls_Manager::MEDIA));

        $this->add_control('reviews', array(
            'label' => __('Reviews', 'ai-theme'),
            'type' => Controls_Manager::REPEATER,
            'fields' => $repeater->get_controls(),
            'default' => array(
                array('author' => 'Emily Roberts', 'role' => 'CEO, TechStart', 'comment' => 'This solution has completely transformed our workflow. Highly recommended for any growing business.', 'rating' => 5),
                array('author' => 'David Kim', 'role' => 'Product Manager', 'comment' => 'Intuitive interface and powerful features. Our team productivity has increased significantly.', 'rating' => 5),
                array('author' => 'Sarah Johnson', 'role' => 'Freelancer', 'comment' => 'Best investment I\'ve made for my business this year. Support team is amazing too!', 'rating' => 5),
            ),
        ));

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $widget_id = $this->get_id();
        $autoplay = $settings['autoplay'] === 'yes';
        $speed = intval($settings['autoplay_speed']) ?: 5000;
        ?>
        <section class="py-20 bg-gradient-to-b from-[var(--bg-secondary)] to-[var(--bg-primary)]" data-theme="<?php echo esc_attr($theme); ?>" data-block-type="reviews" data-block-variant="reviews-slider">
            <div class="max-w-4xl mx-auto px-6">
                <div class="text-center mb-12">
                    <?php if (!empty($settings['heading'])) : ?>
                        <h2 class="text-3xl md:text-4xl font-bold text-[var(--text-primary)] mb-4"><?php echo esc_html($settings['heading']); ?></h2>
                    <?php endif; ?>
                    <?php if (!empty($settings['subheading'])) : ?>
                        <p class="text-lg text-[var(--text-secondary)] max-w-2xl mx-auto"><?php echo esc_html($settings['subheading']); ?></p>
                    <?php endif; ?>
                </div>

                <!-- Slider -->
                <div class="reviews-slider-<?php echo esc_attr($widget_id); ?> relative">
                    <div class="overflow-hidden">
                        <div class="slider-track flex transition-transform duration-500">
                            <?php foreach ($settings['reviews'] as $idx => $review) : ?>
                                <div class="slider-slide flex-shrink-0 w-full px-4">
                                    <div class="bg-[var(--bg-primary)] rounded-[var(--card-radius,20px)] p-8 md:p-12 shadow-xl text-center">
                                        <!-- Quote -->
                                        <svg class="w-12 h-12 text-[var(--primary)] opacity-20 mx-auto mb-6" fill="currentColor" viewBox="0 0 24 24"><path d="M14.017 21v-7.391c0-5.704 3.731-9.57 8.983-10.609l.995 2.151c-2.432.917-3.995 3.638-3.995 5.849h4v10h-9.983zm-14.017 0v-7.391c0-5.704 3.748-9.57 9-10.609l.996 2.151c-2.433.917-3.996 3.638-3.996 5.849h3.983v10h-9.983z"/></svg>

                                        <p class="text-lg md:text-xl text-[var(--text-primary)] leading-relaxed mb-8"><?php echo esc_html($review['comment']); ?></p>

                                        <!-- Rating -->
                                        <div class="flex items-center justify-center gap-1 mb-6">
                                            <?php for ($i = 1; $i <= 5; $i++) : ?>
                                                <svg class="w-5 h-5 <?php echo $i <= $review['rating'] ? 'text-yellow-400 fill-yellow-400' : 'text-[var(--border-color)]'; ?>" viewBox="0 0 24 24"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                                            <?php endfor; ?>
                                        </div>

                                        <!-- Author -->
                                        <div class="flex flex-col items-center">
                                            <?php if (!empty($review['avatar']['url'])) : ?>
                                                <img src="<?php echo esc_url($review['avatar']['url']); ?>" alt="<?php echo esc_attr($review['author']); ?>" class="w-16 h-16 rounded-full object-cover mb-3" />
                                            <?php else : ?>
                                                <div class="w-16 h-16 rounded-full bg-[var(--primary)] flex items-center justify-center mb-3">
                                                    <span class="text-white font-bold text-xl"><?php echo esc_html(substr($review['author'], 0, 1)); ?></span>
                                                </div>
                                            <?php endif; ?>
                                            <p class="font-bold text-[var(--text-primary)]"><?php echo esc_html($review['author']); ?></p>
                                            <?php if (!empty($review['role'])) : ?>
                                                <p class="text-sm text-[var(--text-secondary)]"><?php echo esc_html($review['role']); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Dots -->
                    <div class="flex justify-center gap-2 mt-8">
                        <?php foreach ($settings['reviews'] as $idx => $review) : ?>
                            <button class="slider-dot w-3 h-3 rounded-full transition-colors <?php echo $idx === 0 ? 'bg-[var(--primary)]' : 'bg-[var(--border-color)]'; ?>" data-index="<?php echo esc_attr($idx); ?>"></button>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </section>

        <script>
        (function() {
            const slider = document.querySelector('.reviews-slider-<?php echo esc_attr($widget_id); ?>');
            if (!slider) return;
            
            const track = slider.querySelector('.slider-track');
            const dots = slider.querySelectorAll('.slider-dot');
            const slides = slider.querySelectorAll('.slider-slide');
            let currentIndex = 0;
            const slideCount = slides.length;
            
            function goToSlide(index) {
                currentIndex = index;
                track.style.transform = `translateX(-${index * 100}%)`;
                dots.forEach((dot, i) => {
                    dot.classList.toggle('bg-[var(--primary)]', i === index);
                    dot.classList.toggle('bg-[var(--border-color)]', i !== index);
                });
            }
            
            dots.forEach((dot, i) => {
                dot.addEventListener('click', () => goToSlide(i));
            });
            
            <?php if ($autoplay) : ?>
            setInterval(() => {
                goToSlide((currentIndex + 1) % slideCount);
            }, <?php echo esc_js($speed); ?>);
            <?php endif; ?>
        })();
        </script>
        <?php
    }
}
