<?php
/**
 * Stats Minimal Widget
 *
 * Refined, elegant statistics section with maximum whitespace
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

class Stats_Minimal extends AI_Widget_Base {

    public function get_name() {
        return 'ai_stats_minimal';
    }

    public function get_title() {
        return __('Stats Minimal', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-number-field';
    }

    public function get_categories() {
        return array('ai-theme-stats');
    }

    public function get_keywords() {
        return array('stats', 'statistics', 'numbers', 'minimal', 'clean', 'elegant', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'   => __('Heading', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'   => __('Subheading', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'value',
            array(
                'label'   => __('Value', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'label',
            array(
                'label'   => __('Label', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'stats',
            array(
                'label'   => __('Stats', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(),
            )
        );

        $this->end_controls_section();

        // Style Section
        $this->start_controls_section(
            'style_section',
            array(
                'label' => __('Style', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'value_size',
            array(
                'label'   => __('Value Size', 'ai-theme'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'large',
                'options' => array(
                    'small'  => __('Small', 'ai-theme'),
                    'medium' => __('Medium', 'ai-theme'),
                    'large'  => __('Large', 'ai-theme'),
                ),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        $stats = $settings['stats'];
        $stats_count = count($stats);

        // Column classes
        $cols = min($stats_count, 4);

        // Value size classes
        $value_sizes = array(
            'small'  => 'text-3xl md:text-4xl lg:text-5xl',
            'medium' => 'text-4xl md:text-5xl lg:text-6xl',
            'large'  => 'text-5xl md:text-6xl lg:text-7xl',
        );
        $value_size_class = $value_sizes[$settings['value_size']] ?? $value_sizes['large'];
        ?>
        <section
            class="ai-stats ai-stats-minimal py-24 md:py-32 section-light"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="stats"
            data-block-variant="stats-minimal"
        >
            <div class="max-w-6xl mx-auto px-6">
                <?php if (!empty($settings['heading']) || !empty($settings['subheading'])) : ?>
                    <div class="text-center mb-20">
                        <?php if (!empty($settings['heading'])) : ?>
                            <h2 class="text-3xl md:text-4xl font-light tracking-tight text-[var(--text-primary)] mb-4">
                                <?php echo esc_html($settings['heading']); ?>
                            </h2>
                        <?php endif; ?>
                        <?php if (!empty($settings['subheading'])) : ?>
                            <p class="text-lg text-[var(--text-muted)] font-light max-w-2xl mx-auto">
                                <?php echo esc_html($settings['subheading']); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <!-- Stats Grid -->
                <div class="grid grid-cols-2 md:grid-cols-<?php echo esc_attr($cols); ?> gap-8 md:gap-4">
                    <?php foreach ($stats as $idx => $stat) :
                        $has_border = $idx < $stats_count - 1;
                    ?>
                        <div class="text-center py-8 <?php echo $has_border ? 'md:border-r md:border-[var(--border-color,#e5e7eb)]' : ''; ?>">
                            <!-- Number - The Hero -->
                            <div class="<?php echo esc_attr($value_size_class); ?> font-extralight tracking-tighter text-[var(--text-primary)] mb-3">
                                <?php echo esc_html($stat['value']); ?>
                            </div>

                            <!-- Label -->
                            <div class="text-sm uppercase tracking-[0.2em] text-[var(--text-muted)] font-medium">
                                <?php echo esc_html($stat['label']); ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Bottom Line -->
                <div class="mt-20 flex justify-center">
                    <div class="w-24 h-px bg-[var(--border-color,#e5e7eb)]"></div>
                </div>
            </div>
        </section>
        <?php
    }
}
