<?php
/**
 * Stats Split Widget
 *
 * Asymmetric split-screen design with editorial feel
 *
 * @package AI_Theme
 */

namespace AITheme\Elementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit;
}

class Stats_Split extends AI_Widget_Base {

    public function get_name() {
        return 'ai_stats_split';
    }

    public function get_title() {
        return __('Stats Split', 'ai-theme');
    }

    public function get_icon() {
        return 'eicon-number-field';
    }

    public function get_categories() {
        return array('ai-theme-stats');
    }

    public function get_keywords() {
        return array('stats', 'statistics', 'numbers', 'split', 'asymmetric', 'editorial', 'ai');
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'ai-theme'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'eyebrow',
            array(
                'label'   => __('Eyebrow Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'heading',
            array(
                'label'   => __('Heading', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'subheading',
            array(
                'label'   => __('Subheading', 'ai-theme'),
                'type'    => Controls_Manager::TEXTAREA,
                'default' => '',
            )
        );

        $this->add_control(
            'cta_text',
            array(
                'label'   => __('CTA Text', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'cta_link',
            array(
                'label' => __('CTA Link', 'ai-theme'),
                'type'  => Controls_Manager::URL,
            )
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'value',
            array(
                'label'   => __('Value', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $repeater->add_control(
            'label',
            array(
                'label'   => __('Label', 'ai-theme'),
                'type'    => Controls_Manager::TEXT,
                'default' => '',
            )
        );

        $this->add_control(
            'stats',
            array(
                'label'   => __('Stats', 'ai-theme'),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => array(),
            )
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $theme = get_option('ai_theme_style', 'modern');
        ?>
        <section
            class="ai-stats ai-stats-split min-h-[80vh] grid grid-cols-1 lg:grid-cols-2"
            data-theme="<?php echo esc_attr($theme); ?>" data-block-type="stats"
            data-block-variant="stats-split"
        >
            <!-- Left Side - Statement -->
            <div class="relative section-secondary p-10 md:p-16 lg:p-20 flex flex-col justify-center">
                <!-- Decorative Elements -->
                <div class="absolute top-8 left-8 w-20 h-20 border-2 border-[var(--border-color)] rounded-full"></div>
                <div class="absolute bottom-8 right-8 w-32 h-1 bg-[var(--primary)]"></div>

                <div class="relative">
                    <?php if (!empty($settings['eyebrow'])) : ?>
                        <span class="inline-block text-[var(--primary)] text-sm font-bold uppercase tracking-[0.3em] mb-6">
                            <?php echo esc_html($settings['eyebrow']); ?>
                        </span>
                    <?php endif; ?>

                    <?php if (!empty($settings['heading'])) : ?>
                        <h2 class="text-4xl md:text-5xl lg:text-6xl font-bold leading-[1.1] mb-8 text-[var(--text-primary)]" style="font-family: Georgia, serif;">
                            <?php echo esc_html($settings['heading']); ?>
                        </h2>
                    <?php endif; ?>

                    <?php if (!empty($settings['subheading'])) : ?>
                        <p class="text-lg text-[var(--text-secondary)] max-w-md leading-relaxed">
                            <?php echo esc_html($settings['subheading']); ?>
                        </p>
                    <?php endif; ?>

                    <?php if (!empty($settings['cta_text'])) : ?>
                        <div class="mt-10 flex items-center gap-3 group cursor-pointer">
                            <a
                                href="<?php echo esc_url($settings['cta_link']['url'] ?? '#'); ?>"
                                class="text-sm uppercase tracking-wider font-semibold text-[var(--text-primary)] hover:text-[var(--primary)] transition-colors"
                            >
                                <?php echo esc_html($settings['cta_text']); ?>
                            </a>
                            <svg
                                class="w-4 h-4 text-[var(--primary)] group-hover:translate-x-1 transition-transform"
                                fill="none"
                                viewBox="0 0 24 24"
                                stroke="currentColor"
                            >
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8l4 4m0 0l-4 4m4-4H3" />
                            </svg>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Right Side - Stats Grid -->
            <div class="section-light p-10 md:p-16 lg:p-20 flex items-center">
                <div class="w-full grid grid-cols-2 gap-x-12 gap-y-16">
                    <?php foreach ($settings['stats'] as $idx => $stat) : ?>
                        <div class="relative group">
                            <!-- Hover Accent -->
                            <div class="absolute -left-4 top-0 bottom-0 w-1 bg-[var(--primary)] scale-y-0 group-hover:scale-y-100 origin-top transition-transform duration-300"></div>

                            <!-- Number -->
                            <div class="text-5xl md:text-6xl font-bold text-[var(--text-primary)] mb-3 tracking-tight">
                                <?php echo esc_html($stat['value']); ?>
                            </div>

                            <!-- Label -->
                            <div class="text-[var(--text-muted)] font-medium">
                                <?php echo esc_html($stat['label']); ?>
                            </div>

                            <!-- Index Number -->
                            <div class="absolute -top-2 -right-2 text-8xl font-bold text-[var(--text-primary)] opacity-10 select-none -z-10">
                                <?php echo str_pad($idx + 1, 2, '0', STR_PAD_LEFT); ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php
    }
}
